import { motion } from "framer-motion";
import { MapPin, Bed, Bath, Square, Heart, ArrowUpRight } from "lucide-react";
import { Button } from "@/components/ui/button";
import { cn } from "@/lib/utils";

interface PropertyCardProps {
  image: string;
  title: string;
  location: string;
  price: string;
  beds: number;
  baths: number;
  sqft: string;
  builder: string;
  isNew?: boolean;
  isFeatured?: boolean;
  index?: number;
}

export function PropertyCard({
  image,
  title,
  location,
  price,
  beds,
  baths,
  sqft,
  builder,
  isNew = false,
  isFeatured = false,
  index = 0,
}: PropertyCardProps) {
  return (
    <motion.div
      initial={{ opacity: 0, y: 30 }}
      whileInView={{ opacity: 1, y: 0 }}
      viewport={{ once: true }}
      transition={{ duration: 0.5, delay: index * 0.1 }}
      className={cn(
        "group relative bg-card rounded-2xl overflow-hidden hover-lift",
        isFeatured && "lg:col-span-2 lg:row-span-2"
      )}
    >
      {/* Image */}
      <div className={cn("relative overflow-hidden", isFeatured ? "h-80 lg:h-full" : "h-64")}>
        <img
          src={image}
          alt={title}
          className="w-full h-full object-cover transition-transform duration-500 group-hover:scale-110"
        />
        <div className="absolute inset-0 bg-gradient-to-t from-navy/80 via-navy/20 to-transparent" />

        {/* Badges */}
        <div className="absolute top-4 left-4 flex gap-2">
          {isNew && (
            <span className="px-3 py-1 rounded-full text-xs font-semibold gradient-gold text-navy">
              New Launch
            </span>
          )}
          {isFeatured && (
            <span className="px-3 py-1 rounded-full text-xs font-semibold bg-navy text-cream">
              Featured
            </span>
          )}
        </div>

        {/* Favorite Button */}
        <button className="absolute top-4 right-4 w-10 h-10 rounded-full glass-card flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity duration-300 hover:bg-white">
          <Heart className="w-5 h-5 text-navy" />
        </button>

        {/* Price Tag */}
        <div className="absolute bottom-4 left-4">
          <div className="text-2xl font-bold text-cream">{price}</div>
          <div className="text-sm text-cream/80">Starting Price</div>
        </div>

        {/* Quick View Button */}
        <div className="absolute bottom-4 right-4 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
          <Button variant="glass" size="sm">
            Quick View
            <ArrowUpRight className="w-4 h-4" />
          </Button>
        </div>
      </div>

      {/* Content */}
      <div className="p-5 space-y-4">
        <div>
          <h3 className="text-lg font-semibold text-navy group-hover:text-gold transition-colors duration-200">
            {title}
          </h3>
          <div className="flex items-center gap-1 text-sm text-muted-foreground mt-1">
            <MapPin className="w-4 h-4" />
            {location}
          </div>
        </div>

        {/* Features */}
        <div className="flex items-center gap-4 text-sm text-muted-foreground">
          <div className="flex items-center gap-1.5">
            <Bed className="w-4 h-4 text-gold" />
            <span>{beds} Beds</span>
          </div>
          <div className="flex items-center gap-1.5">
            <Bath className="w-4 h-4 text-gold" />
            <span>{baths} Baths</span>
          </div>
          <div className="flex items-center gap-1.5">
            <Square className="w-4 h-4 text-gold" />
            <span>{sqft}</span>
          </div>
        </div>

        {/* Builder */}
        <div className="flex items-center justify-between pt-4 border-t border-border">
          <div className="text-sm">
            <span className="text-muted-foreground">by </span>
            <span className="font-medium text-navy">{builder}</span>
          </div>
          <Button variant="gold" size="sm">
            View Details
          </Button>
        </div>
      </div>
    </motion.div>
  );
}
