import { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Menu, X, Building2, User, Shield, ChevronDown } from "lucide-react";
import { Button } from "@/components/ui/button";
import { cn } from "@/lib/utils";

const navLinks = [
  { label: "Properties", href: "#properties" },
  { label: "Builders", href: "#builders" },
  { label: "About", href: "#about" },
  { label: "Contact", href: "#contact" },
];

const panelLinks = [
  { label: "Builder Portal", icon: Building2, href: "/builder" },
  { label: "Buyer Dashboard", icon: User, href: "/dashboard" },
  { label: "Admin Panel", icon: Shield, href: "/admin" },
];

export function Navbar() {
  const [isOpen, setIsOpen] = useState(false);
  const [showPanels, setShowPanels] = useState(false);

  return (
    <motion.header
      initial={{ y: -100 }}
      animate={{ y: 0 }}
      transition={{ duration: 0.6, ease: "easeOut" }}
      className="fixed top-0 left-0 right-0 z-50"
    >
      <div className="mx-4 mt-4">
        <nav className="glass-card rounded-2xl px-6 py-4 max-w-7xl mx-auto">
          <div className="flex items-center justify-between">
            {/* Logo */}
            <a href="/" className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-xl gradient-gold flex items-center justify-center shadow-gold">
                <Building2 className="w-5 h-5 text-navy" />
              </div>
              <div className="flex flex-col">
                <span className="text-xl font-bold text-navy tracking-tight">Λꜱᴛʀᴏɴɪx ʀᴇᴀʟᴛʏ</span>
                <span className="text-[10px] text-muted-foreground uppercase tracking-widest">Premium Realty</span>
              </div>
            </a>

            {/* Desktop Navigation */}
            <div className="hidden lg:flex items-center gap-8">
              {navLinks.map((link) => (
                <a
                  key={link.label}
                  href={link.href}
                  className="text-sm font-medium text-muted-foreground hover:text-navy transition-colors duration-200"
                >
                  {link.label}
                </a>
              ))}

              {/* Panels Dropdown */}
              <div className="relative">
                <button
                  onClick={() => setShowPanels(!showPanels)}
                  className="flex items-center gap-1 text-sm font-medium text-muted-foreground hover:text-navy transition-colors duration-200"
                >
                  Portals
                  <ChevronDown className={cn("w-4 h-4 transition-transform duration-200", showPanels && "rotate-180")} />
                </button>

                <AnimatePresence>
                  {showPanels && (
                    <motion.div
                      initial={{ opacity: 0, y: 10 }}
                      animate={{ opacity: 1, y: 0 }}
                      exit={{ opacity: 0, y: 10 }}
                      transition={{ duration: 0.2 }}
                      className="absolute top-full right-0 mt-2 w-56 glass-card rounded-xl p-2 shadow-lg"
                    >
                      {panelLinks.map((panel) => (
                        <a
                          key={panel.label}
                          href={panel.href}
                          className="flex items-center gap-3 px-4 py-3 rounded-lg hover:bg-secondary transition-colors duration-200"
                        >
                          <panel.icon className="w-5 h-5 text-gold" />
                          <span className="text-sm font-medium text-navy">{panel.label}</span>
                        </a>
                      ))}
                    </motion.div>
                  )}
                </AnimatePresence>
              </div>
            </div>

            {/* CTA Buttons */}
            <div className="hidden lg:flex items-center gap-3">
              <Button variant="ghost" size="sm">
                Sign In
              </Button>
              <Button variant="gold" size="default">
                Get Started
              </Button>
            </div>

            {/* Mobile Menu Button */}
            <button
              onClick={() => setIsOpen(!isOpen)}
              className="lg:hidden p-2 rounded-lg hover:bg-secondary transition-colors"
            >
              {isOpen ? <X className="w-6 h-6" /> : <Menu className="w-6 h-6" />}
            </button>
          </div>

          {/* Mobile Menu */}
          <AnimatePresence>
            {isOpen && (
              <motion.div
                initial={{ height: 0, opacity: 0 }}
                animate={{ height: "auto", opacity: 1 }}
                exit={{ height: 0, opacity: 0 }}
                transition={{ duration: 0.3 }}
                className="lg:hidden overflow-hidden"
              >
                <div className="pt-6 pb-4 space-y-4">
                  {navLinks.map((link) => (
                    <a
                      key={link.label}
                      href={link.href}
                      className="block text-base font-medium text-muted-foreground hover:text-navy transition-colors"
                    >
                      {link.label}
                    </a>
                  ))}
                  <div className="pt-4 border-t border-border space-y-2">
                    {panelLinks.map((panel) => (
                      <a
                        key={panel.label}
                        href={panel.href}
                        className="flex items-center gap-3 py-2 text-muted-foreground hover:text-navy"
                      >
                        <panel.icon className="w-5 h-5 text-gold" />
                        <span className="text-sm font-medium">{panel.label}</span>
                      </a>
                    ))}
                  </div>
                  <div className="pt-4 flex flex-col gap-3">
                    <Button variant="outline" className="w-full">
                      Sign In
                    </Button>
                    <Button variant="gold" className="w-full">
                      Get Started
                    </Button>
                  </div>
                </div>
              </motion.div>
            )}
          </AnimatePresence>
        </nav>
      </div>
    </motion.header>
  );
}
