import { useState } from "react";
import { motion } from "framer-motion";
import {
  Building2,
  Users,
  BarChart3,
  Settings,
  Shield,
  Bot,
  FileText,
  Bell,
  Search,
  TrendingUp,
  TrendingDown,
  Activity,
  UserCheck,
  Home,
  ChevronDown,
  MoreVertical,
} from "lucide-react";
import { Button } from "@/components/ui/button";

const sidebarItems = [
  { icon: BarChart3, label: "Overview", active: true },
  { icon: Building2, label: "Builders" },
  { icon: Users, label: "Users" },
  { icon: Home, label: "Properties" },
  { icon: Bot, label: "AI Settings" },
  { icon: FileText, label: "Content" },
  { icon: Shield, label: "Moderation" },
  { icon: Settings, label: "Settings" },
];

const stats = [
  { label: "Total Users", value: "12,456", change: "+12%", up: true, icon: Users },
  { label: "Active Builders", value: "58", change: "+5", up: true, icon: Building2 },
  { label: "Properties Listed", value: "524", change: "+23", up: true, icon: Home },
  { label: "Leads Generated", value: "3,847", change: "+34%", up: true, icon: TrendingUp },
];

const recentActivity = [
  { action: "New builder registered", subject: "Oberoi Realty", time: "2 min ago", type: "builder" },
  { action: "Property approved", subject: "Skyline Residences", time: "15 min ago", type: "property" },
  { action: "Lead assigned", subject: "Rahul → Agent Priya", time: "1 hour ago", type: "lead" },
  { action: "AI content generated", subject: "5 blog posts", time: "2 hours ago", type: "ai" },
  { action: "New user signup", subject: "150 users today", time: "3 hours ago", type: "user" },
];

const topBuilders = [
  { name: "Prestige Group", properties: 24, leads: 456, revenue: "₹12.5 Cr" },
  { name: "Godrej Properties", properties: 18, leads: 324, revenue: "₹8.2 Cr" },
  { name: "Lodha Group", properties: 15, leads: 287, revenue: "₹6.8 Cr" },
  { name: "Oberoi Realty", properties: 12, leads: 198, revenue: "₹5.4 Cr" },
];

const AdminDashboard = () => {
  return (
    <div className="min-h-screen bg-background flex">
      {/* Sidebar */}
      <motion.aside
        initial={{ x: -250 }}
        animate={{ x: 0 }}
        className="w-64 gradient-navy flex flex-col fixed h-full z-40"
      >
        {/* Logo */}
        <div className="p-6 border-b border-white/10">
          <a href="/" className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-xl gradient-gold flex items-center justify-center shadow-gold">
              <Building2 className="w-5 h-5 text-navy" />
            </div>
            <div className="flex flex-col">
              <span className="text-lg font-bold text-cream tracking-tight">Λꜱᴛʀᴏɴɪx ʀᴇᴀʟᴛʏ</span>
              <span className="text-[9px] text-cream/50 uppercase tracking-widest">Admin Panel</span>
            </div>
          </a>
        </div>

        {/* Nav Items */}
        <nav className="flex-1 p-4 space-y-1">
          {sidebarItems.map((item) => (
            <button
              key={item.label}
              className={`w-full flex items-center gap-3 px-4 py-3 rounded-xl text-sm font-medium transition-all duration-200 ${
                item.active
                  ? "bg-white/10 text-gold"
                  : "text-cream/60 hover:bg-white/5 hover:text-cream"
              }`}
            >
              <item.icon className="w-5 h-5" />
              {item.label}
            </button>
          ))}
        </nav>

        {/* Admin User */}
        <div className="p-4 border-t border-white/10">
          <div className="flex items-center gap-3 p-3 rounded-xl bg-white/5">
            <div className="w-10 h-10 rounded-full bg-gold flex items-center justify-center text-navy font-semibold">
              A
            </div>
            <div className="flex-1">
              <div className="text-sm font-medium text-cream">Admin</div>
              <div className="text-xs text-cream/50">Super Admin</div>
            </div>
            <ChevronDown className="w-4 h-4 text-cream/50" />
          </div>
        </div>
      </motion.aside>

      {/* Main Content */}
      <main className="flex-1 ml-64 p-8">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-2xl font-bold text-navy">Admin Dashboard</h1>
            <p className="text-muted-foreground">Platform overview and management.</p>
          </div>
          <div className="flex items-center gap-3">
            <div className="relative">
              <Search className="w-4 h-4 absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground" />
              <input
                type="text"
                placeholder="Search..."
                className="pl-9 pr-4 py-2 rounded-lg border border-border bg-background text-sm focus:outline-none focus:ring-2 focus:ring-gold/50 w-64"
              />
            </div>
            <Button variant="outline" size="icon" className="relative">
              <Bell className="w-5 h-5" />
              <span className="absolute -top-1 -right-1 w-4 h-4 rounded-full bg-destructive text-[10px] text-white flex items-center justify-center">
                5
              </span>
            </Button>
          </div>
        </div>

        {/* Stats */}
        <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          {stats.map((stat, index) => (
            <motion.div
              key={stat.label}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-card rounded-2xl p-6 border border-border hover-lift"
            >
              <div className="flex items-center justify-between mb-4">
                <div className="w-12 h-12 rounded-xl bg-secondary flex items-center justify-center">
                  <stat.icon className="w-6 h-6 text-gold" />
                </div>
                <span
                  className={`text-xs font-medium px-2 py-1 rounded-full flex items-center gap-1 ${
                    stat.up ? "bg-green-50 text-green-600" : "bg-red-50 text-red-600"
                  }`}
                >
                  {stat.up ? <TrendingUp className="w-3 h-3" /> : <TrendingDown className="w-3 h-3" />}
                  {stat.change}
                </span>
              </div>
              <div className="text-2xl font-bold text-navy mb-1">{stat.value}</div>
              <div className="text-sm text-muted-foreground">{stat.label}</div>
            </motion.div>
          ))}
        </div>

        <div className="grid lg:grid-cols-3 gap-6">
          {/* Recent Activity */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4 }}
            className="bg-card rounded-2xl p-6 border border-border"
          >
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-lg font-semibold text-navy flex items-center gap-2">
                <Activity className="w-5 h-5 text-gold" />
                Recent Activity
              </h2>
            </div>
            <div className="space-y-4">
              {recentActivity.map((activity, index) => (
                <div key={index} className="flex items-start gap-3">
                  <div
                    className={`w-2 h-2 rounded-full mt-2 ${
                      activity.type === "builder"
                        ? "bg-blue-500"
                        : activity.type === "property"
                        ? "bg-green-500"
                        : activity.type === "ai"
                        ? "bg-purple-500"
                        : "bg-gold"
                    }`}
                  />
                  <div className="flex-1">
                    <div className="text-sm text-navy">{activity.action}</div>
                    <div className="text-xs text-muted-foreground">{activity.subject}</div>
                  </div>
                  <span className="text-xs text-muted-foreground">{activity.time}</span>
                </div>
              ))}
            </div>
          </motion.div>

          {/* Top Builders */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.5 }}
            className="lg:col-span-2 bg-card rounded-2xl p-6 border border-border"
          >
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-lg font-semibold text-navy flex items-center gap-2">
                <UserCheck className="w-5 h-5 text-gold" />
                Top Performing Builders
              </h2>
              <Button variant="ghost" size="sm">View All</Button>
            </div>
            <div className="overflow-x-auto">
              <table className="w-full">
                <thead>
                  <tr className="border-b border-border">
                    <th className="text-left py-3 text-sm font-medium text-muted-foreground">Builder</th>
                    <th className="text-left py-3 text-sm font-medium text-muted-foreground">Properties</th>
                    <th className="text-left py-3 text-sm font-medium text-muted-foreground">Leads</th>
                    <th className="text-left py-3 text-sm font-medium text-muted-foreground">Revenue</th>
                    <th className="text-right py-3 text-sm font-medium text-muted-foreground">Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {topBuilders.map((builder, index) => (
                    <tr key={index} className="border-b border-border last:border-0 hover:bg-secondary/50">
                      <td className="py-4">
                        <div className="flex items-center gap-3">
                          <div className="w-10 h-10 rounded-full gradient-gold flex items-center justify-center text-navy font-semibold">
                            {builder.name.charAt(0)}
                          </div>
                          <span className="font-medium text-navy">{builder.name}</span>
                        </div>
                      </td>
                      <td className="py-4 text-muted-foreground">{builder.properties}</td>
                      <td className="py-4 text-muted-foreground">{builder.leads}</td>
                      <td className="py-4 font-medium text-gold">{builder.revenue}</td>
                      <td className="py-4 text-right">
                        <Button variant="ghost" size="icon">
                          <MoreVertical className="w-4 h-4" />
                        </Button>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
            </div>
          </motion.div>
        </div>

        {/* AI Status */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.6 }}
          className="mt-6 bg-gradient-to-r from-navy to-navy-light rounded-2xl p-6 text-cream"
        >
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-4">
              <div className="w-14 h-14 rounded-xl gradient-gold flex items-center justify-center">
                <Bot className="w-7 h-7 text-navy" />
              </div>
              <div>
                <h3 className="text-lg font-semibold">AI System Status</h3>
                <p className="text-cream/70 text-sm">All systems operational • 99.9% uptime</p>
              </div>
            </div>
            <div className="flex items-center gap-6">
              <div className="text-center">
                <div className="text-2xl font-bold">1,247</div>
                <div className="text-xs text-cream/60">Chats Today</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold">89%</div>
                <div className="text-xs text-cream/60">Resolution Rate</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold">2.3s</div>
                <div className="text-xs text-cream/60">Avg Response</div>
              </div>
              <Button variant="gold">Configure AI</Button>
            </div>
          </div>
        </motion.div>
      </main>
    </div>
  );
};

export default AdminDashboard;
