import { useState } from "react";
import { motion } from "framer-motion";
import {
  Building2,
  Home,
  Users,
  BarChart3,
  Plus,
  Bell,
  Settings,
  ChevronDown,
  TrendingUp,
  Eye,
  MessageSquare,
  Calendar,
  ArrowUpRight,
  Search,
  Filter,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import property1 from "@/assets/property-1.jpg";
import property2 from "@/assets/property-2.jpg";
import property3 from "@/assets/property-3.jpg";

const sidebarItems = [
  { icon: Home, label: "Dashboard", active: true },
  { icon: Building2, label: "Properties" },
  { icon: Users, label: "Leads" },
  { icon: BarChart3, label: "Analytics" },
  { icon: MessageSquare, label: "AI Content" },
  { icon: Calendar, label: "Visits" },
  { icon: Settings, label: "Settings" },
];

const stats = [
  { label: "Total Properties", value: "24", change: "+3", icon: Building2, color: "bg-blue-500" },
  { label: "Active Leads", value: "156", change: "+23", icon: Users, color: "bg-green-500" },
  { label: "This Month Views", value: "12.4K", change: "+18%", icon: Eye, color: "bg-gold" },
  { label: "Conversion Rate", value: "3.2%", change: "+0.4%", icon: TrendingUp, color: "bg-purple-500" },
];

const recentLeads = [
  { name: "Rahul Sharma", property: "Skyline Residences", status: "Hot", time: "2h ago" },
  { name: "Priya Patel", property: "Ocean View Towers", status: "Warm", time: "4h ago" },
  { name: "Amit Singh", property: "Green Valley Villas", status: "New", time: "6h ago" },
  { name: "Neha Gupta", property: "Skyline Residences", status: "Hot", time: "1d ago" },
];

const properties = [
  { image: property1, title: "Skyline Residences", leads: 45, views: 1200, status: "Active" },
  { image: property2, title: "Ocean View Towers", leads: 32, views: 890, status: "Active" },
  { image: property3, title: "Green Valley Villas", leads: 28, views: 650, status: "Draft" },
];

const BuilderDashboard = () => {
  const [sidebarOpen, setSidebarOpen] = useState(true);

  return (
    <div className="min-h-screen bg-background flex">
      {/* Sidebar */}
      <motion.aside
        initial={{ x: -250 }}
        animate={{ x: 0 }}
        className="w-64 bg-card border-r border-border flex flex-col fixed h-full z-40"
      >
        {/* Logo */}
        <div className="p-6 border-b border-border">
          <a href="/" className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-xl gradient-gold flex items-center justify-center shadow-gold">
              <Building2 className="w-5 h-5 text-navy" />
            </div>
            <div className="flex flex-col">
              <span className="text-lg font-bold text-navy tracking-tight">Λꜱᴛʀᴏɴɪx ʀᴇᴀʟᴛʏ</span>
              <span className="text-[9px] text-muted-foreground uppercase tracking-widest">Builder Portal</span>
            </div>
          </a>
        </div>

        {/* Nav Items */}
        <nav className="flex-1 p-4 space-y-1">
          {sidebarItems.map((item) => (
            <button
              key={item.label}
              className={`w-full flex items-center gap-3 px-4 py-3 rounded-xl text-sm font-medium transition-all duration-200 ${
                item.active
                  ? "bg-gold/10 text-gold"
                  : "text-muted-foreground hover:bg-secondary hover:text-navy"
              }`}
            >
              <item.icon className="w-5 h-5" />
              {item.label}
            </button>
          ))}
        </nav>

        {/* User */}
        <div className="p-4 border-t border-border">
          <div className="flex items-center gap-3 p-3 rounded-xl bg-secondary">
            <div className="w-10 h-10 rounded-full gradient-gold flex items-center justify-center text-navy font-semibold">
              PG
            </div>
            <div className="flex-1">
              <div className="text-sm font-medium text-navy">Prestige Group</div>
              <div className="text-xs text-muted-foreground">Premium Builder</div>
            </div>
            <ChevronDown className="w-4 h-4 text-muted-foreground" />
          </div>
        </div>
      </motion.aside>

      {/* Main Content */}
      <main className="flex-1 ml-64 p-8">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-2xl font-bold text-navy">Dashboard</h1>
            <p className="text-muted-foreground">Welcome back! Here's your property overview.</p>
          </div>
          <div className="flex items-center gap-3">
            <Button variant="outline" size="icon" className="relative">
              <Bell className="w-5 h-5" />
              <span className="absolute -top-1 -right-1 w-4 h-4 rounded-full bg-destructive text-[10px] text-white flex items-center justify-center">
                3
              </span>
            </Button>
            <Button variant="gold">
              <Plus className="w-5 h-5" />
              Add Property
            </Button>
          </div>
        </div>

        {/* Stats */}
        <div className="grid sm:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          {stats.map((stat, index) => (
            <motion.div
              key={stat.label}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: index * 0.1 }}
              className="bg-card rounded-2xl p-6 border border-border hover-lift"
            >
              <div className="flex items-center justify-between mb-4">
                <div className={`w-12 h-12 rounded-xl ${stat.color} flex items-center justify-center`}>
                  <stat.icon className="w-6 h-6 text-white" />
                </div>
                <span className="text-xs font-medium text-green-600 bg-green-50 px-2 py-1 rounded-full">
                  {stat.change}
                </span>
              </div>
              <div className="text-2xl font-bold text-navy mb-1">{stat.value}</div>
              <div className="text-sm text-muted-foreground">{stat.label}</div>
            </motion.div>
          ))}
        </div>

        <div className="grid lg:grid-cols-3 gap-6">
          {/* Recent Leads */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4 }}
            className="lg:col-span-1 bg-card rounded-2xl p-6 border border-border"
          >
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-lg font-semibold text-navy">Recent Leads</h2>
              <Button variant="ghost" size="sm">
                View All
                <ArrowUpRight className="w-4 h-4" />
              </Button>
            </div>
            <div className="space-y-4">
              {recentLeads.map((lead, index) => (
                <div key={index} className="flex items-center gap-4 p-3 rounded-xl hover:bg-secondary transition-colors">
                  <div className="w-10 h-10 rounded-full bg-gold/10 flex items-center justify-center text-gold font-semibold">
                    {lead.name.split(" ").map((n) => n[0]).join("")}
                  </div>
                  <div className="flex-1">
                    <div className="text-sm font-medium text-navy">{lead.name}</div>
                    <div className="text-xs text-muted-foreground">{lead.property}</div>
                  </div>
                  <div className="text-right">
                    <span
                      className={`text-xs font-medium px-2 py-1 rounded-full ${
                        lead.status === "Hot"
                          ? "bg-red-100 text-red-600"
                          : lead.status === "Warm"
                          ? "bg-orange-100 text-orange-600"
                          : "bg-blue-100 text-blue-600"
                      }`}
                    >
                      {lead.status}
                    </span>
                    <div className="text-xs text-muted-foreground mt-1">{lead.time}</div>
                  </div>
                </div>
              ))}
            </div>
          </motion.div>

          {/* Properties */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.5 }}
            className="lg:col-span-2 bg-card rounded-2xl p-6 border border-border"
          >
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-lg font-semibold text-navy">Your Properties</h2>
              <div className="flex items-center gap-2">
                <div className="relative">
                  <Search className="w-4 h-4 absolute left-3 top-1/2 -translate-y-1/2 text-muted-foreground" />
                  <input
                    type="text"
                    placeholder="Search..."
                    className="pl-9 pr-4 py-2 rounded-lg border border-border bg-background text-sm focus:outline-none focus:ring-2 focus:ring-gold/50"
                  />
                </div>
                <Button variant="outline" size="sm">
                  <Filter className="w-4 h-4" />
                </Button>
              </div>
            </div>
            <div className="space-y-4">
              {properties.map((property, index) => (
                <div
                  key={index}
                  className="flex items-center gap-4 p-4 rounded-xl border border-border hover:border-gold/30 transition-colors"
                >
                  <img
                    src={property.image}
                    alt={property.title}
                    className="w-20 h-20 rounded-xl object-cover"
                  />
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-1">
                      <h3 className="font-semibold text-navy">{property.title}</h3>
                      <span
                        className={`text-xs px-2 py-0.5 rounded-full ${
                          property.status === "Active"
                            ? "bg-green-100 text-green-600"
                            : "bg-gray-100 text-gray-600"
                        }`}
                      >
                        {property.status}
                      </span>
                    </div>
                    <div className="flex items-center gap-6 text-sm text-muted-foreground">
                      <span className="flex items-center gap-1">
                        <Users className="w-4 h-4" />
                        {property.leads} leads
                      </span>
                      <span className="flex items-center gap-1">
                        <Eye className="w-4 h-4" />
                        {property.views} views
                      </span>
                    </div>
                  </div>
                  <div className="flex gap-2">
                    <Button variant="outline" size="sm">Edit</Button>
                    <Button variant="gold" size="sm">View Leads</Button>
                  </div>
                </div>
              ))}
            </div>
          </motion.div>
        </div>
      </main>
    </div>
  );
};

export default BuilderDashboard;
