import { useState } from "react";
import { motion } from "framer-motion";
import {
  Heart,
  MapPin,
  Bed,
  Bath,
  Square,
  Calendar,
  MessageSquare,
  Bell,
  Search,
  Filter,
  User,
  Building2,
  Sparkles,
  ChevronDown,
  ArrowRight,
} from "lucide-react";
import { Button } from "@/components/ui/button";
import property1 from "@/assets/property-1.jpg";
import property2 from "@/assets/property-2.jpg";
import property3 from "@/assets/property-3.jpg";

const sidebarItems = [
  { icon: Sparkles, label: "Recommendations", active: true },
  { icon: Heart, label: "Saved Properties" },
  { icon: MessageSquare, label: "Inquiries" },
  { icon: Calendar, label: "Site Visits" },
  { icon: User, label: "Profile" },
];

const savedProperties = [
  {
    image: property1,
    title: "Prestige Ocean View",
    location: "Bandra West, Mumbai",
    price: "₹4.5 Cr",
    beds: 4,
    baths: 3,
    sqft: "2,450 sqft",
    match: 95,
  },
  {
    image: property2,
    title: "Godrej Sky Gardens",
    location: "Worli, Mumbai",
    price: "₹3.2 Cr",
    beds: 3,
    baths: 2,
    sqft: "1,850 sqft",
    match: 88,
  },
];

const recommendations = [
  {
    image: property3,
    title: "Lodha The Park",
    location: "Lower Parel, Mumbai",
    price: "₹2.8 Cr",
    beds: 3,
    baths: 2,
    sqft: "1,650 sqft",
    match: 92,
    reason: "Matches your budget & location preference",
  },
  {
    image: property1,
    title: "Oberoi Esquire",
    location: "Goregaon East, Mumbai",
    price: "₹2.2 Cr",
    beds: 3,
    baths: 2,
    sqft: "1,400 sqft",
    match: 85,
    reason: "Similar to properties you've viewed",
  },
];

const upcomingVisits = [
  { property: "Prestige Ocean View", date: "Jan 22, 2026", time: "10:00 AM", agent: "Raj Kumar" },
  { property: "Godrej Sky Gardens", date: "Jan 25, 2026", time: "2:00 PM", agent: "Priya Singh" },
];

const BuyerDashboard = () => {
  return (
    <div className="min-h-screen bg-background flex">
      {/* Sidebar */}
      <motion.aside
        initial={{ x: -250 }}
        animate={{ x: 0 }}
        className="w-64 bg-card border-r border-border flex flex-col fixed h-full z-40"
      >
        {/* Logo */}
        <div className="p-6 border-b border-border">
          <a href="/" className="flex items-center gap-3">
            <div className="w-10 h-10 rounded-xl gradient-gold flex items-center justify-center shadow-gold">
              <Building2 className="w-5 h-5 text-navy" />
            </div>
            <div className="flex flex-col">
              <span className="text-lg font-bold text-navy tracking-tight">Λꜱᴛʀᴏɴɪx ʀᴇᴀʟᴛʏ</span>
              <span className="text-[9px] text-muted-foreground uppercase tracking-widest">My Dashboard</span>
            </div>
          </a>
        </div>

        {/* Nav Items */}
        <nav className="flex-1 p-4 space-y-1">
          {sidebarItems.map((item) => (
            <button
              key={item.label}
              className={`w-full flex items-center gap-3 px-4 py-3 rounded-xl text-sm font-medium transition-all duration-200 ${
                item.active
                  ? "bg-gold/10 text-gold"
                  : "text-muted-foreground hover:bg-secondary hover:text-navy"
              }`}
            >
              <item.icon className="w-5 h-5" />
              {item.label}
            </button>
          ))}
        </nav>

        {/* User */}
        <div className="p-4 border-t border-border">
          <div className="flex items-center gap-3 p-3 rounded-xl bg-secondary">
            <div className="w-10 h-10 rounded-full gradient-gold flex items-center justify-center text-navy font-semibold">
              RS
            </div>
            <div className="flex-1">
              <div className="text-sm font-medium text-navy">Rahul Sharma</div>
              <div className="text-xs text-muted-foreground">Premium Member</div>
            </div>
            <ChevronDown className="w-4 h-4 text-muted-foreground" />
          </div>
        </div>
      </motion.aside>

      {/* Main Content */}
      <main className="flex-1 ml-64 p-8">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-2xl font-bold text-navy">Welcome, Rahul! 👋</h1>
            <p className="text-muted-foreground">Here are your personalized property recommendations.</p>
          </div>
          <div className="flex items-center gap-3">
            <Button variant="outline" size="icon" className="relative">
              <Bell className="w-5 h-5" />
              <span className="absolute -top-1 -right-1 w-4 h-4 rounded-full bg-destructive text-[10px] text-white flex items-center justify-center">
                2
              </span>
            </Button>
            <Button variant="gold">
              <Search className="w-5 h-5" />
              Search Properties
            </Button>
          </div>
        </div>

        <div className="grid lg:grid-cols-3 gap-6">
          {/* AI Recommendations */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="lg:col-span-2 space-y-6"
          >
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Sparkles className="w-5 h-5 text-gold" />
                <h2 className="text-lg font-semibold text-navy">AI Recommendations</h2>
              </div>
              <Button variant="ghost" size="sm">
                <Filter className="w-4 h-4" />
                Filters
              </Button>
            </div>

            <div className="grid gap-4">
              {recommendations.map((property, index) => (
                <motion.div
                  key={index}
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1 }}
                  className="bg-card rounded-2xl p-4 border border-border hover:border-gold/30 transition-all group"
                >
                  <div className="flex gap-4">
                    <div className="relative">
                      <img
                        src={property.image}
                        alt={property.title}
                        className="w-40 h-32 rounded-xl object-cover"
                      />
                      <div className="absolute top-2 left-2 px-2 py-1 rounded-full bg-gold text-navy text-xs font-semibold">
                        {property.match}% Match
                      </div>
                    </div>
                    <div className="flex-1">
                      <div className="flex items-start justify-between mb-2">
                        <div>
                          <h3 className="font-semibold text-navy group-hover:text-gold transition-colors">
                            {property.title}
                          </h3>
                          <div className="flex items-center gap-1 text-sm text-muted-foreground">
                            <MapPin className="w-3 h-3" />
                            {property.location}
                          </div>
                        </div>
                        <button className="p-2 rounded-full hover:bg-secondary transition-colors">
                          <Heart className="w-5 h-5 text-muted-foreground hover:text-red-500" />
                        </button>
                      </div>
                      <div className="text-xl font-bold text-gold mb-2">{property.price}</div>
                      <div className="flex items-center gap-4 text-sm text-muted-foreground mb-3">
                        <span className="flex items-center gap-1">
                          <Bed className="w-4 h-4" />
                          {property.beds} Beds
                        </span>
                        <span className="flex items-center gap-1">
                          <Bath className="w-4 h-4" />
                          {property.baths} Baths
                        </span>
                        <span className="flex items-center gap-1">
                          <Square className="w-4 h-4" />
                          {property.sqft}
                        </span>
                      </div>
                      <div className="flex items-center justify-between">
                        <span className="text-xs text-muted-foreground bg-secondary px-3 py-1 rounded-full">
                          ✨ {property.reason}
                        </span>
                        <Button variant="gold" size="sm">
                          View Details
                          <ArrowRight className="w-4 h-4" />
                        </Button>
                      </div>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          </motion.div>

          {/* Right Sidebar */}
          <motion.div
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ delay: 0.3 }}
            className="space-y-6"
          >
            {/* Saved Properties */}
            <div className="bg-card rounded-2xl p-6 border border-border">
              <div className="flex items-center justify-between mb-4">
                <h3 className="font-semibold text-navy flex items-center gap-2">
                  <Heart className="w-4 h-4 text-red-500" />
                  Saved Properties
                </h3>
                <span className="text-xs text-muted-foreground">{savedProperties.length} saved</span>
              </div>
              <div className="space-y-3">
                {savedProperties.map((property, index) => (
                  <div key={index} className="flex gap-3 p-2 rounded-lg hover:bg-secondary transition-colors cursor-pointer">
                    <img
                      src={property.image}
                      alt={property.title}
                      className="w-16 h-16 rounded-lg object-cover"
                    />
                    <div>
                      <h4 className="text-sm font-medium text-navy">{property.title}</h4>
                      <p className="text-xs text-muted-foreground">{property.location}</p>
                      <p className="text-sm font-semibold text-gold">{property.price}</p>
                    </div>
                  </div>
                ))}
              </div>
            </div>

            {/* Upcoming Visits */}
            <div className="bg-card rounded-2xl p-6 border border-border">
              <div className="flex items-center justify-between mb-4">
                <h3 className="font-semibold text-navy flex items-center gap-2">
                  <Calendar className="w-4 h-4 text-gold" />
                  Upcoming Visits
                </h3>
              </div>
              <div className="space-y-3">
                {upcomingVisits.map((visit, index) => (
                  <div key={index} className="p-3 rounded-xl bg-secondary/50">
                    <div className="font-medium text-navy text-sm">{visit.property}</div>
                    <div className="text-xs text-muted-foreground mt-1">
                      {visit.date} at {visit.time}
                    </div>
                    <div className="text-xs text-muted-foreground">Agent: {visit.agent}</div>
                  </div>
                ))}
              </div>
              <Button variant="outline" className="w-full mt-4" size="sm">
                Schedule New Visit
              </Button>
            </div>
          </motion.div>
        </div>
      </main>
    </div>
  );
};

export default BuyerDashboard;
