import { useState } from 'react';
import { useLocation, Link } from 'react-router-dom';
import { motion, AnimatePresence } from 'framer-motion';
import { 
  ChevronDown, 
  Settings, 
  Menu, 
  Sun, 
  Moon, 
  Palette,
  Sparkles,
  X
} from 'lucide-react';
import { cn } from '@/lib/utils';
import { navigationData } from '@/data/navigation';
import { getIcon } from '@/lib/icons';
import { useTheme } from '@/hooks/useTheme';

interface DashboardSidebarProps {
  onOpenSettings: () => void;
}

const themeColors = [
  { id: 'cyan', label: 'Cyan', color: 'hsl(174, 100%, 45%)' },
  { id: 'magenta', label: 'Magenta', color: 'hsl(320, 100%, 60%)' },
  { id: 'purple', label: 'Purple', color: 'hsl(270, 100%, 60%)' },
  { id: 'emerald', label: 'Emerald', color: 'hsl(160, 100%, 40%)' },
  { id: 'orange', label: 'Orange', color: 'hsl(25, 100%, 55%)' },
] as const;

export function DashboardSidebar({ onOpenSettings }: DashboardSidebarProps) {
  const location = useLocation();
  const { theme, themeColor, toggleTheme, setThemeColor } = useTheme();
  const [expandedGroups, setExpandedGroups] = useState<string[]>(['dashboard']);
  const [isCollapsed, setIsCollapsed] = useState(false);
  const [showColorPicker, setShowColorPicker] = useState(false);

  const toggleGroup = (groupId: string) => {
    setExpandedGroups(prev =>
      prev.includes(groupId)
        ? prev.filter(id => id !== groupId)
        : [...prev, groupId]
    );
  };

  const isActive = (path: string) => location.pathname === path;

  return (
    <motion.aside
      initial={false}
      animate={{ width: isCollapsed ? 72 : 300 }}
      transition={{ duration: 0.3, ease: 'easeInOut' }}
      className="fixed left-0 top-0 h-screen bg-sidebar sidebar-gradient border-r border-sidebar-border z-50 flex flex-col shadow-xl"
    >
      {/* Header */}
      <div className="h-20 flex items-center justify-between px-4 border-b border-sidebar-border">
        <AnimatePresence mode="wait">
          {!isCollapsed && (
            <motion.div
              initial={{ opacity: 0, x: -20 }}
              animate={{ opacity: 1, x: 0 }}
              exit={{ opacity: 0, x: -20 }}
              transition={{ duration: 0.2 }}
              className="flex items-center gap-3"
            >
              <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-primary to-secondary flex items-center justify-center shadow-lg">
                <Sparkles size={20} className="text-primary-foreground" />
              </div>
              <div>
                <span className="font-display text-lg font-bold text-foreground">AdminHub</span>
                <p className="text-xs text-muted-foreground">Management Portal</p>
              </div>
            </motion.div>
          )}
        </AnimatePresence>
        
        {isCollapsed && (
          <div className="w-10 h-10 rounded-xl bg-gradient-to-br from-primary to-secondary flex items-center justify-center shadow-lg mx-auto">
            <Sparkles size={20} className="text-primary-foreground" />
          </div>
        )}
        
        {!isCollapsed && (
          <button
            onClick={() => setIsCollapsed(true)}
            className="p-2 rounded-lg hover:bg-sidebar-accent text-muted-foreground hover:text-foreground transition-colors"
          >
            <X size={18} />
          </button>
        )}
      </div>

      {/* Collapsed Toggle */}
      {isCollapsed && (
        <button
          onClick={() => setIsCollapsed(false)}
          className="mx-auto mt-4 p-2 rounded-lg hover:bg-sidebar-accent text-muted-foreground hover:text-foreground transition-colors"
        >
          <Menu size={20} />
        </button>
      )}

      {/* Theme Controls */}
      <div className={cn(
        "px-4 py-4 border-b border-sidebar-border",
        isCollapsed && "px-2"
      )}>
        <div className={cn(
          "flex items-center gap-2",
          isCollapsed ? "flex-col" : "justify-between"
        )}>
          {/* Dark/Light Toggle */}
          <motion.button
            whileHover={{ scale: 1.05 }}
            whileTap={{ scale: 0.95 }}
            onClick={toggleTheme}
            className={cn(
              "flex items-center gap-2 px-3 py-2 rounded-xl transition-all",
              "bg-muted hover:bg-muted/80",
              isCollapsed && "p-2"
            )}
          >
            {theme === 'dark' ? (
              <Sun size={18} className="text-primary" />
            ) : (
              <Moon size={18} className="text-primary" />
            )}
            {!isCollapsed && (
              <span className="text-sm font-medium text-foreground">
                {theme === 'dark' ? 'Light' : 'Dark'}
              </span>
            )}
          </motion.button>

          {/* Color Picker Toggle */}
          <div className="relative">
            <motion.button
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
              onClick={() => setShowColorPicker(!showColorPicker)}
              className={cn(
                "flex items-center gap-2 px-3 py-2 rounded-xl transition-all",
                "bg-primary/10 hover:bg-primary/20",
                isCollapsed && "p-2"
              )}
            >
              <Palette size={18} className="text-primary" />
              {!isCollapsed && (
                <span className="text-sm font-medium text-foreground">Theme</span>
              )}
            </motion.button>

            {/* Color Picker Dropdown */}
            <AnimatePresence>
              {showColorPicker && (
                <motion.div
                  initial={{ opacity: 0, y: -10, scale: 0.95 }}
                  animate={{ opacity: 1, y: 0, scale: 1 }}
                  exit={{ opacity: 0, y: -10, scale: 0.95 }}
                  className={cn(
                    "absolute top-full mt-2 p-3 rounded-xl bg-card border border-border shadow-xl z-50",
                    isCollapsed ? "left-0" : "right-0"
                  )}
                >
                  <p className="text-xs font-medium text-muted-foreground mb-2">Choose Color</p>
                  <div className="flex gap-2">
                    {themeColors.map((color) => (
                      <button
                        key={color.id}
                        onClick={() => {
                          setThemeColor(color.id);
                          setShowColorPicker(false);
                        }}
                        className={cn(
                          "w-8 h-8 rounded-full transition-all ring-2 ring-offset-2 ring-offset-card",
                          themeColor === color.id ? "ring-foreground scale-110" : "ring-transparent hover:scale-105"
                        )}
                        style={{ backgroundColor: color.color }}
                        title={color.label}
                      />
                    ))}
                  </div>
                </motion.div>
              )}
            </AnimatePresence>
          </div>
        </div>
      </div>

      {/* Navigation */}
      <nav className="flex-1 overflow-y-auto py-4 scrollbar-thin">
        {navigationData.map(group => {
          const GroupIcon = getIcon(group.icon);
          const isExpanded = expandedGroups.includes(group.id);
          const hasActiveChild = group.items.some(item => isActive(item.path));

          return (
            <div key={group.id} className="mb-1 px-3">
              <button
                onClick={() => !isCollapsed && toggleGroup(group.id)}
                className={cn(
                  "w-full flex items-center gap-3 px-3 py-3 rounded-xl text-left transition-all",
                  "hover:bg-sidebar-accent group",
                  (isExpanded || hasActiveChild) && "bg-sidebar-accent",
                  isCollapsed && "justify-center px-2"
                )}
                title={isCollapsed ? group.title : undefined}
              >
                <div className={cn(
                  "p-2 rounded-lg transition-colors",
                  (isExpanded || hasActiveChild) ? "bg-primary/20" : "bg-muted"
                )}>
                  <GroupIcon
                    size={18}
                    className={cn(
                      "transition-colors",
                      (isExpanded || hasActiveChild) ? "text-primary" : "text-muted-foreground"
                    )}
                  />
                </div>
                <AnimatePresence>
                  {!isCollapsed && (
                    <motion.span
                      initial={{ opacity: 0 }}
                      animate={{ opacity: 1 }}
                      exit={{ opacity: 0 }}
                      className={cn(
                        "flex-1 text-sm font-medium",
                        (isExpanded || hasActiveChild) ? "text-foreground" : "text-muted-foreground"
                      )}
                    >
                      {group.title}
                    </motion.span>
                  )}
                </AnimatePresence>
                {!isCollapsed && (
                  <motion.div
                    animate={{ rotate: isExpanded ? 180 : 0 }}
                    transition={{ duration: 0.2 }}
                  >
                    <ChevronDown size={16} className="text-muted-foreground" />
                  </motion.div>
                )}
              </button>

              <AnimatePresence>
                {isExpanded && !isCollapsed && (
                  <motion.div
                    initial={{ height: 0, opacity: 0 }}
                    animate={{ height: 'auto', opacity: 1 }}
                    exit={{ height: 0, opacity: 0 }}
                    transition={{ duration: 0.2 }}
                    className="overflow-hidden ml-4 mt-1 border-l-2 border-border"
                  >
                    {group.items.map(item => {
                      const ItemIcon = getIcon(item.icon);
                      const active = isActive(item.path);

                      return (
                        <Link
                          key={item.id}
                          to={item.path}
                          className={cn(
                            "flex items-center gap-3 pl-4 pr-3 py-2.5 text-sm transition-all rounded-r-lg ml-1",
                            "hover:bg-sidebar-accent hover:border-l-2 hover:border-primary hover:-ml-0.5",
                            active && "bg-primary/10 border-l-2 border-primary -ml-0.5 text-primary font-medium"
                          )}
                        >
                          <ItemIcon size={16} className={active ? "text-primary" : "text-muted-foreground"} />
                          <span className={active ? "text-primary" : "text-sidebar-foreground"}>
                            {item.title}
                          </span>
                          {item.badge && (
                            <span className="ml-auto px-2 py-0.5 text-xs rounded-full bg-secondary text-secondary-foreground font-medium">
                              {item.badge}
                            </span>
                          )}
                        </Link>
                      );
                    })}
                  </motion.div>
                )}
              </AnimatePresence>
            </div>
          );
        })}
      </nav>

      {/* Settings Button */}
      <div className="p-4 border-t border-sidebar-border">
        <motion.button
          whileHover={{ scale: 1.02 }}
          whileTap={{ scale: 0.98 }}
          onClick={onOpenSettings}
          className={cn(
            "w-full flex items-center gap-3 px-4 py-3 rounded-xl",
            "bg-gradient-to-r from-primary/10 to-secondary/10",
            "hover:from-primary/20 hover:to-secondary/20 transition-all",
            "border border-primary/20",
            isCollapsed && "justify-center px-2"
          )}
        >
          <Settings size={20} className="text-primary" />
          <AnimatePresence>
            {!isCollapsed && (
              <motion.div
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                exit={{ opacity: 0 }}
                className="text-left"
              >
                <span className="text-sm font-medium text-foreground block">Customize</span>
                <span className="text-xs text-muted-foreground">Dashboard Layout</span>
              </motion.div>
            )}
          </AnimatePresence>
        </motion.button>
      </div>
    </motion.aside>
  );
}
