import { useState } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { X, Eye, EyeOff, RotateCcw, GripVertical } from 'lucide-react';
import { Widget } from '@/types/dashboard';

interface SettingsModalProps {
  isOpen: boolean;
  onClose: () => void;
  widgets: Widget[];
  onToggleVisibility: (widgetId: string) => void;
  onReset: () => void;
}

export function SettingsModal({
  isOpen,
  onClose,
  widgets,
  onToggleVisibility,
  onReset,
}: SettingsModalProps) {
  return (
    <AnimatePresence>
      {isOpen && (
        <>
          {/* Backdrop */}
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            className="fixed inset-0 bg-background/80 backdrop-blur-sm z-50"
            onClick={onClose}
          />

          {/* Modal */}
          <motion.div
            initial={{ opacity: 0, scale: 0.9, y: 20 }}
            animate={{ opacity: 1, scale: 1, y: 0 }}
            exit={{ opacity: 0, scale: 0.9, y: 20 }}
            className="fixed top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 z-50 w-full max-w-lg"
          >
            <div className="glass-card neon-border-cyan p-6">
              <div className="flex items-center justify-between mb-6">
                <h2 className="font-display text-xl font-semibold text-foreground">
                  Customize Dashboard
                </h2>
                <button
                  onClick={onClose}
                  className="p-2 rounded-lg hover:bg-muted transition-colors text-muted-foreground hover:text-foreground"
                >
                  <X size={20} />
                </button>
              </div>

              <p className="text-sm text-muted-foreground mb-4">
                Toggle widgets on/off and drag to reorder them on the dashboard.
              </p>

              <div className="space-y-2 max-h-80 overflow-y-auto scrollbar-thin">
                {widgets.map((widget) => (
                  <motion.div
                    key={widget.id}
                    layout
                    className="flex items-center gap-3 p-3 rounded-lg bg-muted/50 hover:bg-muted transition-colors"
                  >
                    <GripVertical size={16} className="text-muted-foreground cursor-grab" />
                    <div className="flex-1">
                      <p className="text-sm font-medium text-foreground">{widget.title}</p>
                      <p className="text-xs text-muted-foreground capitalize">
                        {widget.size} • {widget.type}
                      </p>
                    </div>
                    <button
                      onClick={() => onToggleVisibility(widget.id)}
                      className={`p-2 rounded-lg transition-colors ${
                        widget.visible
                          ? 'bg-neon-cyan/20 text-neon-cyan'
                          : 'bg-muted text-muted-foreground'
                      }`}
                    >
                      {widget.visible ? <Eye size={18} /> : <EyeOff size={18} />}
                    </button>
                  </motion.div>
                ))}
              </div>

              <div className="flex justify-between mt-6 pt-4 border-t border-border">
                <button
                  onClick={onReset}
                  className="flex items-center gap-2 px-4 py-2 rounded-lg text-muted-foreground hover:text-foreground hover:bg-muted transition-colors"
                >
                  <RotateCcw size={16} />
                  <span className="text-sm">Reset to Default</span>
                </button>
                <button
                  onClick={onClose}
                  className="px-6 py-2 rounded-lg bg-neon-cyan text-primary-foreground font-medium hover:bg-neon-cyan/90 transition-colors"
                >
                  Done
                </button>
              </div>
            </div>
          </motion.div>
        </>
      )}
    </AnimatePresence>
  );
}
