import { motion } from 'framer-motion';
import { useSortable } from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';
import { GripVertical } from 'lucide-react';
import { Widget } from '@/types/dashboard';
import { StatsWidget } from './StatsWidget';
import { EventsWidget } from './EventsWidget';
import { NotificationsWidget } from './NotificationsWidget';
import { ActionsWidget } from './ActionsWidget';
import { ChartWidget } from './ChartWidget';
import { ListWidget } from './ListWidget';
import { RevenueWidget } from './RevenueWidget';
import { TicketsWidget } from './TicketsWidget';

interface DraggableWidgetProps {
  widget: Widget;
  isEditing: boolean;
}

const widgetComponents: Record<string, React.FC> = {
  stats: StatsWidget,
  chart: ChartWidget,
  list: ListWidget,
  actions: ActionsWidget,
  events: EventsWidget,
  notifications: NotificationsWidget,
  revenue: RevenueWidget,
  tickets: TicketsWidget,
};

const sizeClasses: Record<string, string> = {
  small: 'col-span-1',
  medium: 'col-span-1 md:col-span-2',
  large: 'col-span-1 md:col-span-2 lg:col-span-4',
};

export function DraggableWidget({ widget, isEditing }: DraggableWidgetProps) {
  const {
    attributes,
    listeners,
    setNodeRef,
    transform,
    transition,
    isDragging,
  } = useSortable({ id: widget.id });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
  };

  const WidgetComponent = widgetComponents[widget.type];

  return (
    <motion.div
      ref={setNodeRef}
      style={style}
      className={`${sizeClasses[widget.size]} relative ${isDragging ? 'z-50' : ''}`}
      initial={{ opacity: 0, scale: 0.9 }}
      animate={{ opacity: 1, scale: 1 }}
      layout
    >
      {isEditing && (
        <div
          {...attributes}
          {...listeners}
          className="absolute -top-2 -left-2 z-10 p-1.5 rounded-lg bg-neon-cyan cursor-grab active:cursor-grabbing"
        >
          <GripVertical size={16} className="text-primary-foreground" />
        </div>
      )}
      <div className={`h-full ${isDragging ? 'opacity-50' : ''}`}>
        {WidgetComponent && <WidgetComponent />}
      </div>
    </motion.div>
  );
}
