import { useState, useEffect, createContext, useContext } from 'react';

type Theme = 'light' | 'dark';
type ThemeColor = 'cyan' | 'magenta' | 'purple' | 'emerald' | 'orange';

interface ThemeContextType {
  theme: Theme;
  themeColor: ThemeColor;
  setTheme: (theme: Theme) => void;
  setThemeColor: (color: ThemeColor) => void;
  toggleTheme: () => void;
}

const ThemeContext = createContext<ThemeContextType | undefined>(undefined);

export function ThemeProvider({ children }: { children: React.ReactNode }) {
  const [theme, setThemeState] = useState<Theme>(() => {
    const stored = localStorage.getItem('dashboard-theme');
    return (stored as Theme) || 'dark';
  });

  const [themeColor, setThemeColorState] = useState<ThemeColor>(() => {
    const stored = localStorage.getItem('dashboard-theme-color');
    return (stored as ThemeColor) || 'cyan';
  });

  useEffect(() => {
    const root = document.documentElement;
    root.classList.remove('light', 'dark');
    root.classList.add(theme);
    localStorage.setItem('dashboard-theme', theme);
  }, [theme]);

  useEffect(() => {
    const root = document.documentElement;
    root.classList.remove('theme-cyan', 'theme-magenta', 'theme-purple', 'theme-emerald', 'theme-orange');
    root.classList.add(`theme-${themeColor}`);
    localStorage.setItem('dashboard-theme-color', themeColor);
  }, [themeColor]);

  const setTheme = (newTheme: Theme) => setThemeState(newTheme);
  const setThemeColor = (color: ThemeColor) => setThemeColorState(color);
  const toggleTheme = () => setThemeState(prev => prev === 'dark' ? 'light' : 'dark');

  return (
    <ThemeContext.Provider value={{ theme, themeColor, setTheme, setThemeColor, toggleTheme }}>
      {children}
    </ThemeContext.Provider>
  );
}

export function useTheme() {
  const context = useContext(ThemeContext);
  if (!context) {
    throw new Error('useTheme must be used within a ThemeProvider');
  }
  return context;
}
