import { useState, useEffect } from 'react';
import { Widget } from '@/types/dashboard';
import { defaultWidgets } from '@/data/widgets';

const STORAGE_KEY = 'dashboard-widget-config';

export function useWidgetConfig() {
  const [widgets, setWidgets] = useState<Widget[]>(() => {
    const stored = localStorage.getItem(STORAGE_KEY);
    if (stored) {
      try {
        return JSON.parse(stored);
      } catch {
        return defaultWidgets;
      }
    }
    return defaultWidgets;
  });

  useEffect(() => {
    localStorage.setItem(STORAGE_KEY, JSON.stringify(widgets));
  }, [widgets]);

  const updateWidgetOrder = (newOrder: Widget[]) => {
    setWidgets(newOrder.map((w, i) => ({ ...w, order: i })));
  };

  const toggleWidgetVisibility = (widgetId: string) => {
    setWidgets(prev =>
      prev.map(w => (w.id === widgetId ? { ...w, visible: !w.visible } : w))
    );
  };

  const resetToDefault = () => {
    setWidgets(defaultWidgets);
  };

  const visibleWidgets = widgets.filter(w => w.visible).sort((a, b) => a.order - b.order);

  return {
    widgets,
    visibleWidgets,
    updateWidgetOrder,
    toggleWidgetVisibility,
    resetToDefault,
  };
}
