import { useState } from 'react';
import { motion } from 'framer-motion';
import {
  DndContext,
  closestCenter,
  KeyboardSensor,
  PointerSensor,
  useSensor,
  useSensors,
  DragEndEvent,
} from '@dnd-kit/core';
import {
  arrayMove,
  SortableContext,
  sortableKeyboardCoordinates,
  rectSortingStrategy,
} from '@dnd-kit/sortable';
import { DashboardSidebar } from '@/components/dashboard/DashboardSidebar';
import { SettingsModal } from '@/components/dashboard/SettingsModal';
import { DraggableWidget } from '@/components/widgets/DraggableWidget';
import { useWidgetConfig } from '@/hooks/useWidgetConfig';
import { Edit3, Check } from 'lucide-react';

const Index = () => {
  const [settingsOpen, setSettingsOpen] = useState(false);
  const [isEditing, setIsEditing] = useState(false);
  const {
    widgets,
    visibleWidgets,
    updateWidgetOrder,
    toggleWidgetVisibility,
    resetToDefault,
  } = useWidgetConfig();

  const sensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;

    if (over && active.id !== over.id) {
      const oldIndex = visibleWidgets.findIndex((w) => w.id === active.id);
      const newIndex = visibleWidgets.findIndex((w) => w.id === over.id);

      const newOrder = arrayMove(visibleWidgets, oldIndex, newIndex);
      
      // Merge with hidden widgets
      const hiddenWidgets = widgets.filter((w) => !w.visible);
      updateWidgetOrder([...newOrder, ...hiddenWidgets]);
    }
  };

  return (
    <div className="min-h-screen bg-background transition-colors duration-300">
      <DashboardSidebar onOpenSettings={() => setSettingsOpen(true)} />
      
      {/* Main Content */}
      <main className="ml-[72px] md:ml-[300px] min-h-screen transition-all duration-300">
        {/* Header */}
        <header className="sticky top-0 z-40 bg-background/80 backdrop-blur-xl border-b border-border">
          <div className="flex items-center justify-between px-8 py-4">
            <div>
              <h1 className="font-display text-2xl font-bold text-foreground">
                Platform Overview
              </h1>
              <p className="text-sm text-muted-foreground">
                Welcome back! Here's what's happening today.
              </p>
            </div>
            <div className="flex items-center gap-3">
              <motion.button
                whileHover={{ scale: 1.05 }}
                whileTap={{ scale: 0.95 }}
                onClick={() => setIsEditing(!isEditing)}
                className={`flex items-center gap-2 px-4 py-2 rounded-lg font-medium transition-all ${
                  isEditing
                    ? 'bg-neon-cyan text-primary-foreground'
                    : 'bg-muted text-foreground hover:bg-muted/80'
                }`}
              >
                {isEditing ? (
                  <>
                    <Check size={18} />
                    <span>Done</span>
                  </>
                ) : (
                  <>
                    <Edit3 size={18} />
                    <span>Edit Layout</span>
                  </>
                )}
              </motion.button>
            </div>
          </div>
        </header>

        {/* Widget Grid */}
        <div className="p-8">
          <DndContext
            sensors={sensors}
            collisionDetection={closestCenter}
            onDragEnd={handleDragEnd}
          >
            <SortableContext
              items={visibleWidgets.map((w) => w.id)}
              strategy={rectSortingStrategy}
            >
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 auto-rows-min">
                {visibleWidgets.map((widget) => (
                  <DraggableWidget
                    key={widget.id}
                    widget={widget}
                    isEditing={isEditing}
                  />
                ))}
              </div>
            </SortableContext>
          </DndContext>

          {visibleWidgets.length === 0 && (
            <motion.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              className="text-center py-20"
            >
              <p className="text-muted-foreground mb-4">No widgets visible</p>
              <button
                onClick={() => setSettingsOpen(true)}
                className="px-6 py-3 rounded-lg bg-neon-cyan text-primary-foreground font-medium hover:bg-neon-cyan/90 transition-colors"
              >
                Customize Dashboard
              </button>
            </motion.div>
          )}
        </div>
      </main>

      {/* Settings Modal */}
      <SettingsModal
        isOpen={settingsOpen}
        onClose={() => setSettingsOpen(false)}
        widgets={widgets}
        onToggleVisibility={toggleWidgetVisibility}
        onReset={resetToDefault}
      />
    </div>
  );
};

export default Index;
