# Troubleshooting Guide

## "Failed to load packages" Error

This error can occur due to several reasons. Follow these steps to diagnose and fix:

### Step 1: Check if Server is Running

**Check server status:**
```bash
# In server directory
cd server
npm run dev
```

You should see:
```
✅ MongoDB Connected: ...
🚀 Server running on port 5000
```

**If server is not running:**
- Make sure you're in the `server` directory
- Check if port 5000 is already in use: `lsof -i :5000`
- Make sure you have a `.env` file in the `server` directory

### Step 2: Check MongoDB Connection

**Verify MongoDB is running:**
```bash
# Check if MongoDB is running
sudo systemctl status mongod

# Or start MongoDB
sudo systemctl start mongod
```

**Check MongoDB connection string in `server/.env`:**
```env
MONGODB_URI=mongodb://localhost:27017/balaji-tour-travel
```

**For MongoDB Atlas:**
```env
MONGODB_URI=mongodb+srv://username:password@cluster.mongodb.net/balaji-tour-travel
```

### Step 3: Seed the Database

**If packages are not loading, you need to seed the database:**
```bash
cd server
npm run seed
```

You should see:
```
✅ 10 Tour packages seeded successfully!
```

### Step 4: Check API URL Configuration

**Frontend API URL (`client/.env`):**
```env
VITE_API_URL=http://localhost:5000/api
```

**Backend CORS (`server/.env`):**
```env
CLIENT_URL=http://localhost:5173
```

### Step 5: Check Browser Console

Open browser DevTools (F12) and check:
1. **Console tab** - Look for error messages
2. **Network tab** - Check if API requests are being made
   - Look for requests to `http://localhost:5000/api/packages`
   - Check if they're failing (red status)

### Step 6: Verify Server Endpoints

**Test the API directly:**
```bash
# Health check
curl http://localhost:5000/api/health

# Get packages
curl http://localhost:5000/api/packages
```

### Common Issues and Solutions

#### Issue 1: "Cannot connect to server"
**Solution:**
- Make sure the server is running on port 5000
- Check if `server/.env` exists and has correct configuration
- Verify no firewall is blocking port 5000

#### Issue 2: "MongoDB connection error"
**Solution:**
- Start MongoDB: `sudo systemctl start mongod`
- Check MongoDB URI in `server/.env`
- For MongoDB Atlas, ensure your IP is whitelisted

#### Issue 3: "No packages found"
**Solution:**
- Seed the database: `cd server && npm run seed`
- Check if packages exist: `curl http://localhost:5000/api/packages`

#### Issue 4: CORS Error
**Solution:**
- Ensure `CLIENT_URL` in `server/.env` matches your frontend URL
- Default should be: `CLIENT_URL=http://localhost:5173`

#### Issue 5: Port Already in Use
**Solution:**
```bash
# Find process using port 5000
lsof -i :5000

# Kill the process
kill -9 <PID>

# Or change port in server/.env
PORT=5001
```

### Quick Diagnostic Commands

```bash
# 1. Check if server is running
curl http://localhost:5000/api/health

# 2. Check if packages endpoint works
curl http://localhost:5000/api/packages

# 3. Check MongoDB connection
cd server
node -e "require('dotenv').config(); const mongoose = require('mongoose'); mongoose.connect(process.env.MONGODB_URI).then(() => console.log('✅ Connected')).catch(e => console.log('❌ Error:', e.message));"
```

### Still Having Issues?

1. **Check server logs** - Look for error messages in the terminal where server is running
2. **Check browser console** - Look for network errors or CORS issues
3. **Verify all environment variables** are set correctly
4. **Restart both server and client** after making changes

---

## Quick Fix Checklist

- [ ] Server is running (`cd server && npm run dev`)
- [ ] MongoDB is running (`sudo systemctl status mongod`)
- [ ] Database is seeded (`cd server && npm run seed`)
- [ ] `server/.env` file exists with correct values
- [ ] `client/.env` file exists with `VITE_API_URL=http://localhost:5000/api`
- [ ] No port conflicts (port 5000 and 5173 are free)
- [ ] Browser console shows no CORS errors

