#!/usr/bin/env node

/**
 * Quick diagnostic script to check if everything is set up correctly
 */

import { fileURLToPath } from 'url';
import { dirname, join } from 'path';
import { existsSync, readFileSync } from 'fs';
import http from 'http';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

console.log('🔍 Checking setup...\n');

let allGood = true;

// Check 1: Server .env file
console.log('1. Checking server/.env file...');
const serverEnvPath = join(__dirname, 'server', '.env');
if (existsSync(serverEnvPath)) {
  console.log('   ✅ server/.env exists');
  const serverEnv = readFileSync(serverEnvPath, 'utf8');
  if (serverEnv.includes('MONGODB_URI')) {
    console.log('   ✅ MONGODB_URI is set');
  } else {
    console.log('   ❌ MONGODB_URI is missing');
    allGood = false;
  }
} else {
  console.log('   ❌ server/.env does not exist');
  console.log('   💡 Create server/.env with required variables (see SETUP.md)');
  allGood = false;
}

// Check 2: Client .env file
console.log('\n2. Checking client/.env file...');
const clientEnvPath = join(__dirname, 'client', '.env');
if (existsSync(clientEnvPath)) {
  console.log('   ✅ client/.env exists');
} else {
  console.log('   ⚠️  client/.env does not exist (optional, will use default)');
}

// Check 3: Server running
console.log('\n3. Checking if server is running on port 5000...');
const checkServer = () => {
  return new Promise((resolve) => {
    const req = http.get('http://localhost:5000/api/health', (res) => {
      if (res.statusCode === 200) {
        console.log('   ✅ Server is running');
        resolve(true);
      } else {
        console.log('   ❌ Server returned error status:', res.statusCode);
        resolve(false);
      }
    });

    req.on('error', () => {
      console.log('   ❌ Server is not running or not accessible');
      console.log('   💡 Start server with: cd server && npm run dev');
      resolve(false);
    });

    req.setTimeout(2000, () => {
      req.destroy();
      console.log('   ❌ Server connection timeout');
      resolve(false);
    });
  });
};

const serverRunning = await checkServer();
if (!serverRunning) {
  allGood = false;
}

// Check 4: Packages endpoint
if (serverRunning) {
  console.log('\n4. Checking packages endpoint...');
  const checkPackages = () => {
    return new Promise((resolve) => {
      const req = http.get('http://localhost:5000/api/packages', (res) => {
        let data = '';
        res.on('data', (chunk) => {
          data += chunk;
        });
        res.on('end', () => {
          try {
            const json = JSON.parse(data);
            if (json.packages && Array.isArray(json.packages)) {
              const count = json.packages.length;
              if (count > 0) {
                console.log(`   ✅ Packages endpoint working (${count} packages found)`);
                resolve(true);
              } else {
                console.log('   ⚠️  Packages endpoint working but no packages found');
                console.log('   💡 Seed database with: cd server && npm run seed');
                resolve(false);
              }
            } else {
              console.log('   ⚠️  Unexpected response format');
              resolve(false);
            }
          } catch (e) {
            console.log('   ❌ Invalid JSON response');
            resolve(false);
          }
        });
      });

      req.on('error', () => {
        console.log('   ❌ Cannot access packages endpoint');
        resolve(false);
      });

      req.setTimeout(2000, () => {
        req.destroy();
        console.log('   ❌ Request timeout');
        resolve(false);
      });
    });
  };

  const packagesOk = await checkPackages();
  if (!packagesOk) {
    allGood = false;
  }
}

// Summary
console.log('\n' + '='.repeat(50));
if (allGood) {
  console.log('✅ All checks passed! Your setup looks good.');
} else {
  console.log('❌ Some issues found. Please fix them before running the app.');
  console.log('\n📖 See TROUBLESHOOTING.md for detailed solutions.');
}
console.log('='.repeat(50));

process.exit(allGood ? 0 : 1);

