import { useState } from 'react';
import { useLocation } from 'react-router-dom';
import { HelpCircle, Phone, Mail, MessageCircle, X } from 'lucide-react';
import { motion, AnimatePresence } from 'framer-motion';

const NeedHelp = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [showText, setShowText] = useState(false);
  const location = useLocation();

  // Hide Need Help on admin pages
  if (location.pathname.startsWith('/admin')) {
    return null;
  }

  const helpOptions = [
    {
      icon: Phone,
      label: 'Call Now',
      value: '+91 9729849875',
      action: () => window.open('tel:9729849875'),
      color: 'text-green-600',
    },
    {
      icon: Mail,
      label: 'Email Us',
      value: 'balajitourandtravels2026@gmail.com',
      action: () => window.open('mailto:balajitourandtravels2026@gmail.com'),
      color: 'text-blue-600',
    },
    {
      icon: MessageCircle,
      label: 'WhatsApp',
      value: '9729849875',
      action: () => window.open('https://wa.me/919729849875'),
      color: 'text-green-500',
    },
  ];

  return (
    <>
      {/* Floating Button */}
      <motion.div
        initial={{ scale: 0 }}
        animate={{ scale: 1 }}
        className="fixed bottom-6 right-6 z-50"
        onMouseEnter={() => setShowText(true)}
        onMouseLeave={() => setShowText(false)}
      >
        <motion.button
          whileHover={{ scale: 1.05 }}
          whileTap={{ scale: 0.95 }}
          onClick={() => setIsOpen(!isOpen)}
          className="flex items-center space-x-3 bg-gradient-to-r from-primary-600 to-primary-700 text-white px-4 py-3 rounded-full shadow-2xl hover:shadow-primary-500/50 transition-all duration-300"
        >
          <AnimatePresence>
            {showText && (
              <motion.span
                initial={{ opacity: 0, width: 0 }}
                animate={{ opacity: 1, width: 'auto' }}
                exit={{ opacity: 0, width: 0 }}
                className="whitespace-nowrap font-semibold overflow-hidden"
              >
                Need help!
              </motion.span>
            )}
          </AnimatePresence>
          <HelpCircle className="w-6 h-6 flex-shrink-0" />
        </motion.button>
      </motion.div>

      {/* Help Panel */}
      <AnimatePresence>
        {isOpen && (
          <>
            <motion.div
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              exit={{ opacity: 0 }}
              onClick={() => setIsOpen(false)}
              className="fixed inset-0 bg-black/50 z-40"
            />
            <motion.div
              initial={{ opacity: 0, y: 20, scale: 0.95 }}
              animate={{ opacity: 1, y: 0, scale: 1 }}
              exit={{ opacity: 0, y: 20, scale: 0.95 }}
              className="fixed bottom-24 right-6 z-50 bg-white rounded-2xl shadow-2xl p-6 w-80"
            >
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-xl font-bold text-gray-900">Need Help?</h3>
                <button
                  onClick={() => setIsOpen(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  <X className="w-5 h-5" />
                </button>
              </div>
              <p className="text-gray-600 text-sm mb-4">
                We're here to assist you! Choose how you'd like to reach us:
              </p>
              <div className="space-y-3">
                {helpOptions.map((option, index) => {
                  const Icon = option.icon;
                  return (
                    <motion.button
                      key={index}
                      initial={{ opacity: 0, x: -20 }}
                      animate={{ opacity: 1, x: 0 }}
                      transition={{ delay: index * 0.1 }}
                      onClick={option.action}
                      className="w-full flex items-center space-x-3 p-3 rounded-lg hover:bg-gray-50 transition-colors border border-gray-200"
                    >
                      <Icon className={`w-5 h-5 ${option.color} flex-shrink-0`} />
                      <div className="text-left flex-1">
                        <p className="font-semibold text-gray-900">{option.label}</p>
                        <p className="text-sm text-gray-600 break-words">{option.value}</p>
                      </div>
                    </motion.button>
                  );
                })}
              </div>
            </motion.div>
          </>
        )}
      </AnimatePresence>
    </>
  );
};

export default NeedHelp;

