import { useEffect, useState } from 'react';
import { useParams, Link } from 'react-router-dom';
import { motion } from 'framer-motion';
import { CheckCircle, XCircle, Clock, Calendar, Users, Phone, Mail } from 'lucide-react';
import { getBookingById, verifyPayment } from '../services/api';
import toast from 'react-hot-toast';

const BookingConfirmation = () => {
  const { bookingId } = useParams();
  const [booking, setBooking] = useState(null);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    fetchBooking();
    const interval = setInterval(() => {
      checkPaymentStatus();
    }, 5000); // Check every 5 seconds

    return () => clearInterval(interval);
  }, [bookingId]);

  const fetchBooking = async () => {
    try {
      const { data } = await getBookingById(bookingId);
      setBooking(data);
    } catch (error) {
      toast.error('Booking not found');
    } finally {
      setLoading(false);
    }
  };

  const checkPaymentStatus = async () => {
    try {
      const { data } = await verifyPayment(bookingId);
      if (data.paymentStatus === 'paid' && booking?.paymentStatus !== 'paid') {
        fetchBooking(); // Refresh booking data
        toast.success('Payment confirmed!');
      }
    } catch (error) {
      console.error('Error checking payment status:', error);
    }
  };

  if (loading) {
    return (
      <div className="pt-20 min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary-600"></div>
      </div>
    );
  }

  if (!booking) {
    return (
      <div className="pt-20 min-h-screen flex items-center justify-center">
        <div className="text-center">
          <XCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
          <h2 className="text-2xl font-bold mb-2">Booking Not Found</h2>
          <Link to="/packages" className="text-primary-600 hover:underline">
            Browse Packages
          </Link>
        </div>
      </div>
    );
  }

  const isPaid = booking.paymentStatus === 'paid';
  const isPending = booking.paymentStatus === 'pending';

  return (
    <div className="pt-20 min-h-screen bg-gray-50 py-12">
      <div className="max-w-3xl mx-auto px-4 sm:px-6 lg:px-8">
        <motion.div
          initial={{ opacity: 0, scale: 0.9 }}
          animate={{ opacity: 1, scale: 1 }}
          className="bg-white rounded-xl shadow-lg p-8 text-center"
        >
          {isPaid ? (
            <>
              <motion.div
                initial={{ scale: 0 }}
                animate={{ scale: 1 }}
                transition={{ type: 'spring', stiffness: 200 }}
              >
                <CheckCircle className="w-20 h-20 text-green-500 mx-auto mb-4" />
              </motion.div>
              <h1 className="text-3xl font-bold mb-2 text-green-600">Payment Successful!</h1>
              <p className="text-gray-600 mb-8">Your booking has been confirmed.</p>
            </>
          ) : isPending ? (
            <>
              <Clock className="w-20 h-20 text-yellow-500 mx-auto mb-4" />
              <h1 className="text-3xl font-bold mb-2 text-yellow-600">Payment Pending</h1>
              <p className="text-gray-600 mb-8">We're waiting for payment confirmation.</p>
            </>
          ) : (
            <>
              <XCircle className="w-20 h-20 text-red-500 mx-auto mb-4" />
              <h1 className="text-3xl font-bold mb-2 text-red-600">Payment Failed</h1>
              <p className="text-gray-600 mb-8">Your payment could not be processed.</p>
            </>
          )}

          {/* Booking Details */}
          <div className="bg-gray-50 rounded-lg p-6 mb-6 text-left">
            <h2 className="text-xl font-bold mb-4">Booking Details</h2>
            <div className="space-y-3">
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Package:</span>
                <span className="font-semibold">{booking.packageName}</span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Booking ID:</span>
                <span className="font-mono text-sm">{booking._id}</span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Travel Date:</span>
                <span className="font-semibold">
                  {new Date(booking.travelDate).toLocaleDateString()}
                </span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Travelers:</span>
                <span className="font-semibold">{booking.numberOfTravelers}</span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Total Amount:</span>
                <span className="font-bold text-primary-600 text-lg">
                  ₹{booking.totalAmount.toLocaleString()}
                </span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-gray-600">Payment Status:</span>
                <span
                  className={`font-semibold ${
                    isPaid
                      ? 'text-green-600'
                      : isPending
                      ? 'text-yellow-600'
                      : 'text-red-600'
                  }`}
                >
                  {booking.paymentStatus.toUpperCase()}
                </span>
              </div>
              {booking.veegahTransactionId && (
                <div className="flex items-center justify-between">
                  <span className="text-gray-600">Transaction ID:</span>
                  <span className="font-mono text-sm">{booking.veegahTransactionId}</span>
                </div>
              )}
            </div>
          </div>

          {/* Contact Info */}
          <div className="bg-primary-50 rounded-lg p-6 mb-6">
            <h3 className="font-semibold mb-3">Need Help?</h3>
            <div className="space-y-2 text-sm">
              <div className="flex items-center justify-center space-x-2">
                <Phone className="w-4 h-4" />
                <a href="tel:9729849875" className="hover:underline">
                  +91 9729849875
                </a>
              </div>
              <div className="flex items-center justify-center space-x-2">
                <Mail className="w-4 h-4" />
                <a
                  href="mailto:balajitourandtravels2026@gmail.com"
                  className="hover:underline"
                >
                  balajitourandtravels2026@gmail.com
                </a>
              </div>
            </div>
          </div>

          {/* Actions */}
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            {isPaid ? (
              <>
                <Link to="/packages" className="btn-secondary">
                  Browse More Packages
                </Link>
                <button
                  onClick={() => window.print()}
                  className="btn-primary"
                >
                  Print Confirmation
                </button>
              </>
            ) : (
              <>
                <Link to="/packages" className="btn-secondary">
                  Try Again
                </Link>
                <Link to="/contact" className="btn-primary">
                  Contact Support
                </Link>
              </>
            )}
          </div>
        </motion.div>
      </div>
    </div>
  );
};

export default BookingConfirmation;

