import { useEffect, useState } from 'react';
import { Link } from 'react-router-dom';
import { motion } from 'framer-motion';
import { ArrowRight, Star, MapPin, Clock, Users, Shield, Award, Headphones, Plane, Globe, Camera, Heart, TrendingUp, CheckCircle, Mail, Phone, Calendar, CreditCard, UserCheck, Sparkles } from 'lucide-react';
import { getFeaturedPackages } from '../services/api';
import toast from 'react-hot-toast';

const Home = () => {
  const [packages, setPackages] = useState([]);
  const [loading, setLoading] = useState(true);
  const [currentImageIndex, setCurrentImageIndex] = useState(0);

  const heroImages = [
    'https://images.unsplash.com/photo-1488646953014-85cb44e25828?w=1920',
    'https://images.unsplash.com/photo-1578662996442-48f60103fc96?w=1920',
    'https://images.unsplash.com/photo-1512343879784-a960bf40e7f2?w=1920',
    'https://images.unsplash.com/photo-1559827260-dc66d52bef19?w=1920',
    'https://images.unsplash.com/photo-1506905925346-21bda4d32df4?w=1920',
    'https://images.unsplash.com/photo-1512453979798-5ea266f8880c?w=1920',
  ];

  useEffect(() => {
    fetchFeaturedPackages();
  }, []);

  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentImageIndex((prevIndex) => (prevIndex + 1) % heroImages.length);
    }, 5000); // Change image every 5 seconds

    return () => clearInterval(interval);
  }, [heroImages.length]);

  const fetchFeaturedPackages = async () => {
    try {
      const { data } = await getFeaturedPackages();
      setPackages(data);
    } catch (error) {
      console.error('Error loading packages:', error);
      const errorMessage = error.response?.data?.message || error.message || 'Failed to load packages';
      toast.error(errorMessage.includes('connect') ? 'Cannot connect to server. Please check if the server is running.' : errorMessage);
    } finally {
      setLoading(false);
    }
  };

  const testimonials = [
    {
      name: 'Rajesh Kumar',
      location: 'Mumbai',
      rating: 5,
      text: 'Amazing experience with Balaji Tour & Travel! The Kashmir trip was beyond expectations. Everything was perfectly organized.',
      image: 'https://ui-avatars.com/api/?name=Rajesh+Kumar&background=0ea5e9&color=fff',
      package: 'Kashmir Paradise',
    },
    {
      name: 'Priya Sharma',
      location: 'Delhi',
      rating: 5,
      text: 'Best travel agency! Professional service and well-planned itinerary. Highly recommended! The team was always available.',
      image: 'https://ui-avatars.com/api/?name=Priya+Sharma&background=0ea5e9&color=fff',
      package: 'Goa Beach Escape',
    },
    {
      name: 'Amit Patel',
      location: 'Ahmedabad',
      rating: 5,
      text: 'Excellent service from booking to completion. The Dubai trip was fantastic! Great value for money.',
      image: 'https://ui-avatars.com/api/?name=Amit+Patel&background=0ea5e9&color=fff',
      package: 'Dubai Delight',
    },
    {
      name: 'Sneha Reddy',
      location: 'Bangalore',
      rating: 5,
      text: 'Outstanding service! The Manali trip exceeded all expectations. Will definitely book again.',
      image: 'https://ui-avatars.com/api/?name=Sneha+Reddy&background=0ea5e9&color=fff',
      package: 'Manali Adventure',
    },
    {
      name: 'Vikram Singh',
      location: 'Pune',
      rating: 5,
      text: 'Smooth booking process and amazing trip to Shimla. The guides were knowledgeable and friendly.',
      image: 'https://ui-avatars.com/api/?name=Vikram+Singh&background=0ea5e9&color=fff',
      package: 'Shimla Hills',
    },
    {
      name: 'Anjali Mehta',
      location: 'Chennai',
      rating: 5,
      text: 'Perfect vacation! The Ooty package was well-planned and the accommodation was excellent.',
      image: 'https://ui-avatars.com/api/?name=Anjali+Mehta&background=0ea5e9&color=fff',
      package: 'Ooty Serenity',
    },
  ];

  const destinations = [
    { name: 'Kashmir', image: 'https://images.unsplash.com/photo-1578662996442-48f60103fc96?w=800', packages: 12 },
    { name: 'Goa', image: 'https://images.unsplash.com/photo-1512343879784-a960bf40e7f2?w=800', packages: 15 },
    { name: 'Manali', image: 'https://images.unsplash.com/photo-1559827260-dc66d52bef19?w=800', packages: 10 },
    { name: 'Dubai', image: 'https://images.unsplash.com/photo-1512453979798-5ea266f8880c?w=800', packages: 8 },
    { name: 'Shimla', image: 'https://images.unsplash.com/photo-1506905925346-21bda4d32df4?w=800&h=600&fit=crop', packages: 9 },
    { name: 'Ooty', image: 'data:image/jpeg;base64,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', packages: 7 },
  ];

  const categories = [
    { name: 'Adventure Tours', icon: '🏔️', count: 25, color: 'from-blue-500 to-cyan-500' },
    { name: 'Beach Holidays', icon: '🏖️', count: 18, color: 'from-teal-500 to-blue-500' },
    { name: 'Hill Stations', icon: '⛰️', count: 22, color: 'from-green-500 to-emerald-500' },
    { name: 'Cultural Tours', icon: '🏛️', count: 15, color: 'from-purple-500 to-pink-500' },
    { name: 'Honeymoon', icon: '💑', count: 12, color: 'from-pink-500 to-rose-500' },
    { name: 'Family Packages', icon: '👨‍👩‍👧‍👦', count: 20, color: 'from-orange-500 to-yellow-500' },
  ];

  const stats = [
    { number: '10,000+', label: 'Happy Travelers', icon: Users },
    { number: '500+', label: 'Destinations', icon: Globe },
    { number: '15+', label: 'Years Experience', icon: Award },
    { number: '98%', label: 'Satisfaction Rate', icon: Star },
  ];

  const howItWorks = [
    { step: 1, title: 'Choose Package', desc: 'Browse our curated collection of amazing travel packages', icon: Globe },
    { step: 2, title: 'Book Online', desc: 'Easy online booking with secure payment gateway', icon: CreditCard },
    { step: 3, title: 'Get Confirmation', desc: 'Receive instant confirmation and travel documents', icon: CheckCircle },
    { step: 4, title: 'Enjoy Trip', desc: 'Experience unforgettable moments with expert guides', icon: Heart },
  ];

  const features = [
    { icon: Shield, title: 'Secure Booking', desc: '100% secure payment gateway' },
    { icon: Award, title: 'Best Prices', desc: 'Competitive pricing guaranteed' },
    { icon: Headphones, title: '24/7 Support', desc: 'Round the clock assistance' },
    { icon: Users, title: 'Expert Guides', desc: 'Professional tour guides' },
  ];

  return (
    <div className="pt-20">
      {/* Hero Section */}
      <section className="relative h-[90vh] flex items-center justify-center overflow-hidden">
        <div className="absolute inset-0 z-0">
          {/* Carousel Background */}
          {heroImages.map((image, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0 }}
              animate={{
                opacity: index === currentImageIndex ? 1 : 0,
                scale: index === currentImageIndex ? 1 : 1.1,
              }}
              transition={{ duration: 1.5, ease: 'easeInOut' }}
              className="absolute inset-0"
            >
              <img
                src={image}
                alt={`Travel destination ${index + 1}`}
                className="w-full h-full object-cover"
              />
            </motion.div>
          ))}
          <div className="absolute inset-0 bg-gradient-to-r from-primary-900/80 to-primary-700/60" />
          
          {/* Carousel Indicators */}
          <div className="absolute bottom-8 left-1/2 transform -translate-x-1/2 z-20 flex gap-2">
            {heroImages.map((_, index) => (
              <button
                key={index}
                onClick={() => setCurrentImageIndex(index)}
                className={`h-2 rounded-full transition-all duration-300 ${
                  index === currentImageIndex
                    ? 'w-8 bg-white'
                    : 'w-2 bg-white/50 hover:bg-white/75'
                }`}
                aria-label={`Go to slide ${index + 1}`}
              />
            ))}
          </div>
        </div>

        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
          className="relative z-10 text-center text-white px-4 max-w-4xl mx-auto"
        >
          <h1 className="text-5xl md:text-7xl font-bold mb-6">
            Discover Your Next
            <span className="block text-accent-300">Adventure</span>
          </h1>
          <p className="text-xl md:text-2xl mb-8 text-gray-200">
            Premium travel experiences crafted for unforgettable memories
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link to="/packages" className="btn-primary text-lg px-8 py-4 inline-flex items-center justify-center">
              View Packages <ArrowRight className="ml-2 w-5 h-5" />
            </Link>
            <Link to="/contact" className="btn-secondary text-lg px-8 py-4 text-white border-white bg-white/10 hover:bg-white/20">
              Contact Us
            </Link>
          </div>
        </motion.div>
      </section>



      {/* Features Section */}
      <section className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            className="text-center mb-16"
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Why Choose Us</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Experience the difference with our premium travel services
            </p>
          </motion.div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {features.map((feature, index) => {
              const Icon = feature.icon;
              return (
                <motion.div
                  key={index}
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ delay: index * 0.1 }}
                  className="group text-center p-8 rounded-2xl bg-gradient-to-br from-gray-50 to-white border border-gray-100 hover:border-primary-300 hover:shadow-xl transition-all duration-300"
                >
                  <div className="inline-flex items-center justify-center w-20 h-20 bg-gradient-to-br from-primary-100 to-primary-200 rounded-2xl mb-6 group-hover:scale-110 transition-transform">
                    <Icon className="w-10 h-10 text-primary-600" />
                  </div>
                  <h3 className="font-bold text-xl mb-3">{feature.title}</h3>
                  <p className="text-gray-600">{feature.desc}</p>
                </motion.div>
              );
            })}
          </div>
        </div>
      </section>

            {/* Statistics Section */}
      <section className="py-20 bg-gradient-to-br from-primary-600 via-primary-700 to-primary-800 text-white relative overflow-hidden">
        <div className="absolute inset-0 opacity-10">
          <div className="absolute top-0 left-0 w-96 h-96 bg-accent-300 rounded-full blur-3xl"></div>
          <div className="absolute bottom-0 right-0 w-96 h-96 bg-accent-400 rounded-full blur-3xl"></div>
        </div>
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative z-10">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-8">
            {stats.map((stat, index) => {
              const Icon = stat.icon;
              return (
                <motion.div
                  key={index}
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ delay: index * 0.1 }}
                  className="text-center"
                >
                  <div className="inline-flex items-center justify-center w-20 h-20 bg-white/10 backdrop-blur-md rounded-full mb-4 border border-white/20">
                    <Icon className="w-10 h-10" />
                  </div>
                  <motion.div
                    initial={{ opacity: 0 }}
                    whileInView={{ opacity: 1 }}
                    viewport={{ once: true }}
                    transition={{ delay: index * 0.1 + 0.2 }}
                    className="text-4xl md:text-5xl font-bold mb-2"
                  >
                    {stat.number}
                  </motion.div>
                  <div className="text-gray-200 text-sm md:text-base">{stat.label}</div>
                </motion.div>
              );
            })}
          </div>
        </div>
      </section>

      {/* Featured Packages */}
      <section className="py-20 bg-gradient-to-b from-white to-gray-50">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            className="text-center mb-16"
          >
            <div className="inline-flex items-center gap-2 bg-primary-100 text-primary-700 px-4 py-2 rounded-full mb-4">
              <Star className="w-5 h-5 fill-current" />
              <span className="font-semibold">Featured</span>
            </div>
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Featured Packages</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Handpicked destinations for your perfect getaway. Curated by travel experts.
            </p>
          </motion.div>

          {loading ? (
            <div className="text-center py-20">
              <div className="inline-block animate-spin rounded-full h-16 w-16 border-4 border-primary-200 border-t-primary-600"></div>
              <p className="mt-4 text-gray-600">Loading amazing packages...</p>
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
              {packages.slice(0, 6).map((pkg, index) => (
                <motion.div
                  key={pkg._id}
                  initial={{ opacity: 0, y: 30 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ delay: index * 0.1 }}
                  className="group bg-white rounded-2xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 transform hover:-translate-y-2"
                >
                  <div className="relative h-64 overflow-hidden">
                    <img
                      src={pkg.images[0]}
                      alt={pkg.title}
                      className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent" />
                    {pkg.featured && (
                      <span className="absolute top-4 left-4 bg-gradient-to-r from-accent-500 to-accent-600 text-white px-4 py-2 rounded-full text-sm font-semibold shadow-lg">
                        ⭐ Featured
                      </span>
                    )}
                    <div className="absolute bottom-4 right-4 bg-white/90 backdrop-blur-sm px-3 py-1 rounded-full">
                      <div className="flex items-center text-yellow-500">
                        <Star className="w-4 h-4 fill-current" />
                        <span className="ml-1 text-sm font-semibold text-gray-800">4.8</span>
                      </div>
                    </div>
                  </div>
                  <div className="p-6">
                    <div className="flex items-center justify-between mb-3">
                      <span className="bg-primary-100 text-primary-700 px-3 py-1 rounded-full text-xs font-semibold">
                        {pkg.category}
                      </span>
                    </div>
                    <h3 className="text-2xl font-bold mb-3 text-gray-900 group-hover:text-primary-600 transition-colors">
                      {pkg.title}
                    </h3>
                    <p className="text-gray-600 text-sm mb-4 line-clamp-2">{pkg.shortDescription}</p>
                    <div className="flex items-center justify-between mb-4 text-sm text-gray-600">
                      <div className="flex items-center">
                        <MapPin className="w-4 h-4 mr-1 text-primary-600" />
                        {pkg.location}
                      </div>
                      <div className="flex items-center">
                        <Clock className="w-4 h-4 mr-1 text-primary-600" />
                        {pkg.duration} Days
                      </div>
                    </div>
                    <div className="flex items-center justify-between mb-6">
                      <div>
                        <span className="text-3xl font-bold text-primary-600">₹{pkg.price.toLocaleString()}</span>
                        {pkg.originalPrice && (
                          <span className="text-gray-400 line-through ml-2 text-lg">₹{pkg.originalPrice.toLocaleString()}</span>
                        )}
                        <div className="text-xs text-gray-500 mt-1">per person</div>
                      </div>
                    </div>
                    <Link
                      to={`/package/${pkg.slug}`}
                      className="btn-primary w-full text-center block group-hover:shadow-lg transition-all"
                    >
                      View Details <ArrowRight className="inline ml-2 w-4 h-4" />
                    </Link>
                  </div>
                </motion.div>
              ))}
            </div>
          )}

          <div className="text-center mt-16">
            <Link to="/packages" className="btn-secondary inline-flex items-center text-lg px-8 py-4 hover:shadow-lg transition-all">
              Explore All Packages <ArrowRight className="ml-2 w-5 h-5" />
            </Link>
          </div>
        </div>
      </section>

      {/* Popular Destinations */}
      <section className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            className="text-center mb-16"
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Popular Destinations</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Explore our most loved destinations around the world
            </p>
          </motion.div>
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-4 md:gap-6">
            {destinations.map((dest, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, scale: 0.9 }}
                whileInView={{ opacity: 1, scale: 1 }}
                viewport={{ once: true }}
                transition={{ delay: index * 0.1 }}
                className="group relative h-48 md:h-64 rounded-2xl overflow-hidden cursor-pointer"
              >
                <img
                  src={dest.image}
                  alt={dest.name}
                  className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-500"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/40 to-transparent" />
                <div className="absolute bottom-0 left-0 right-0 p-4 text-white">
                  <h3 className="font-bold text-lg mb-1">{dest.name}</h3>
                  <p className="text-sm text-gray-200">{dest.packages} Packages</p>
                </div>
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* Travel Categories */}
      <section className="py-20 bg-gradient-to-br from-gray-50 to-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            className="text-center mb-16"
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Travel Categories</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Find the perfect trip type for your travel style
            </p>
          </motion.div>
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-6">
            {categories.map((category, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ delay: index * 0.1 }}
                className="group text-center p-6 rounded-2xl bg-white border-2 border-gray-100 hover:border-primary-300 hover:shadow-xl transition-all cursor-pointer"
              >
                <div className={`text-5xl mb-4 transform group-hover:scale-110 transition-transform`}>
                  {category.icon}
                </div>
                <h3 className="font-bold text-lg mb-2 group-hover:text-primary-600 transition-colors">
                  {category.name}
                </h3>
                <p className="text-sm text-gray-600">{category.count} Packages</p>
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* How It Works */}
      <section className="py-20 bg-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            className="text-center mb-16"
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-4">How It Works</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Book your dream vacation in just 4 simple steps
            </p>
          </motion.div>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {howItWorks.map((step, index) => {
              const Icon = step.icon;
              return (
                <motion.div
                  key={index}
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ delay: index * 0.1 }}
                  className="relative text-center"
                >
                  {index < howItWorks.length - 1 && (
                    <div className="hidden lg:block absolute top-12 left-full w-full h-0.5 bg-gradient-to-r from-primary-300 to-primary-100 transform translate-x-4" />
                  )}
                  <div className="relative inline-flex items-center justify-center w-24 h-24 bg-gradient-to-br from-primary-500 to-primary-600 rounded-full mb-6 shadow-lg">
                    <Icon className="w-12 h-12 text-white" />
                    <div className="absolute -top-2 -right-2 w-8 h-8 bg-accent-500 rounded-full flex items-center justify-center text-white font-bold text-sm">
                      {step.step}
                    </div>
                  </div>
                  <h3 className="text-xl font-bold mb-3">{step.title}</h3>
                  <p className="text-gray-600">{step.desc}</p>
                </motion.div>
              );
            })}
          </div>
        </div>
      </section>



      {/* Testimonials */}
      <section className="py-20 bg-gradient-to-b from-gray-50 to-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <motion.div
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            className="text-center mb-16"
          >
            <div className="inline-flex items-center gap-2 bg-yellow-100 text-yellow-700 px-4 py-2 rounded-full mb-4">
              <Star className="w-5 h-5 fill-current" />
              <span className="font-semibold">4.8/5 Average Rating</span>
            </div>
            <h2 className="text-4xl md:text-5xl font-bold mb-4">What Our Customers Say</h2>
            <p className="text-gray-600 text-lg max-w-2xl mx-auto">
              Real experiences from real travelers who trusted us with their journeys
            </p>
          </motion.div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {testimonials.map((testimonial, index) => (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                whileInView={{ opacity: 1, y: 0 }}
                viewport={{ once: true }}
                transition={{ delay: index * 0.1 }}
                className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-2xl transition-all border border-gray-100"
              >
                <div className="flex items-center mb-6">
                  <img
                    src={testimonial.image}
                    alt={testimonial.name}
                    className="w-16 h-16 rounded-full mr-4 border-2 border-primary-200"
                  />
                  <div>
                    <h4 className="font-bold text-lg">{testimonial.name}</h4>
                    <p className="text-sm text-gray-600">{testimonial.location}</p>
                    <p className="text-xs text-primary-600 font-semibold mt-1">{testimonial.package}</p>
                  </div>
                </div>
                <div className="flex mb-4">
                  {[...Array(testimonial.rating)].map((_, i) => (
                    <Star key={i} className="w-5 h-5 fill-yellow-400 text-yellow-400" />
                  ))}
                </div>
                <p className="text-gray-700 italic leading-relaxed">"{testimonial.text}"</p>
              </motion.div>
            ))}
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-20 bg-gradient-to-r from-primary-600 via-primary-700 to-primary-800 text-white relative overflow-hidden">
        <div className="absolute inset-0 opacity-10">
          <div className="absolute top-0 right-0 w-96 h-96 bg-accent-300 rounded-full blur-3xl"></div>
          <div className="absolute bottom-0 left-0 w-96 h-96 bg-accent-400 rounded-full blur-3xl"></div>
        </div>
        <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center relative z-10">
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            whileInView={{ opacity: 1, y: 0 }}
            viewport={{ once: true }}
          >
            <h2 className="text-4xl md:text-5xl font-bold mb-6">
              Ready to Start Your Journey?
            </h2>
            <p className="text-xl text-gray-200 mb-10 max-w-2xl mx-auto">
              Join thousands of happy travelers and create memories that last a lifetime. Book your dream vacation today!
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Link
                to="/packages"
                className="bg-white text-primary-600 px-10 py-4 rounded-lg font-semibold text-lg hover:bg-gray-100 transition-all transform hover:scale-105 shadow-xl inline-flex items-center justify-center"
              >
                Browse Packages <ArrowRight className="ml-2 w-5 h-5" />
              </Link>
              <Link
                to="/contact"
                className="bg-white/10 backdrop-blur-md border-2 border-white text-white px-10 py-4 rounded-lg font-semibold text-lg hover:bg-white/20 transition-all transform hover:scale-105 inline-flex items-center justify-center"
              >
                Contact Us
              </Link>
            </div>
          </motion.div>
        </div>
      </section>
    </div>
  );
};

export default Home;


