import { useState, useEffect } from 'react';
import { useParams, useNavigate, Link } from 'react-router-dom';
import { motion } from 'framer-motion';
import { QrCode, Copy, Check, ArrowLeft, Smartphone, AlertCircle } from 'lucide-react';
import { getBookingById } from '../services/api';
import toast from 'react-hot-toast';

const UPIPayment = () => {
  const { bookingId } = useParams();
  const navigate = useNavigate();
  const [booking, setBooking] = useState(null);
  const [loading, setLoading] = useState(true);
  const [copied, setCopied] = useState(false);

  // Replace this with your actual UPI ID
  const UPI_ID = 'your-upi-id@paytm'; // Example: 9729849875@paytm or balajitours@ybl
  const PAYEE_NAME = 'Balaji Tour & Travel';

  useEffect(() => {
    fetchBooking();
  }, [bookingId]);

  const fetchBooking = async () => {
    try {
      const { data } = await getBookingById(bookingId);
      setBooking(data.booking || data);
    } catch (error) {
      toast.error('Booking not found');
      navigate('/packages');
    } finally {
      setLoading(false);
    }
  };

  const generateUPIUrl = (amount) => {
    const upiUrl = `upi://pay?pa=${UPI_ID}&pn=${encodeURIComponent(PAYEE_NAME)}&am=${amount}&cu=INR&tn=${encodeURIComponent(`Payment for ${booking?.packageName || 'Tour Package'}`)}`;
    return upiUrl;
  };

  const generateQRCodeUrl = (amount) => {
    const upiUrl = generateUPIUrl(amount);
    return `https://api.qrserver.com/v1/create-qr-code/?size=300x300&data=${encodeURIComponent(upiUrl)}`;
  };

  const handleUPIClick = () => {
    if (booking) {
      const upiUrl = generateUPIUrl(booking.totalAmount);
      window.location.href = upiUrl;
    }
  };

  const handleQRClick = () => {
    if (booking) {
      const upiUrl = generateUPIUrl(booking.totalAmount);
      window.location.href = upiUrl;
    }
  };

  const copyUPIId = () => {
    navigator.clipboard.writeText(UPI_ID);
    setCopied(true);
    toast.success('UPI ID copied to clipboard!');
    setTimeout(() => setCopied(false), 2000);
  };

  if (loading) {
    return (
      <div className="pt-20 min-h-screen flex items-center justify-center">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary-600"></div>
      </div>
    );
  }

  if (!booking) return null;

  return (
    <div className="pt-20 min-h-screen bg-gray-50">
      <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Back Button */}
        <Link
          to="/packages"
          className="inline-flex items-center text-primary-600 hover:text-primary-700 mb-6"
        >
          <ArrowLeft className="w-5 h-5 mr-2" />
          Back to Packages
        </Link>

        {/* Payment Card */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          className="bg-white rounded-2xl shadow-xl p-8"
        >
          {/* Header */}
          <div className="text-center mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Complete Your Payment</h1>
            <p className="text-gray-600">Scan QR code or click to pay via UPI</p>
          </div>

          {/* Booking Summary */}
          <div className="bg-gray-50 rounded-xl p-6 mb-8">
            <h2 className="text-xl font-semibold mb-4">Booking Summary</h2>
            <div className="space-y-2 text-gray-700">
              <div className="flex justify-between">
                <span>Package:</span>
                <span className="font-medium">{booking.packageName}</span>
              </div>
              <div className="flex justify-between">
                <span>Travelers:</span>
                <span className="font-medium">{booking.numberOfTravelers}</span>
              </div>
              <div className="flex justify-between">
                <span>Travel Date:</span>
                <span className="font-medium">
                  {new Date(booking.travelDate).toLocaleDateString('en-IN', {
                    day: 'numeric',
                    month: 'long',
                    year: 'numeric',
                  })}
                </span>
              </div>
              <div className="border-t border-gray-300 pt-2 mt-2">
                <div className="flex justify-between text-lg font-bold">
                  <span>Total Amount:</span>
                  <span className="text-primary-600">₹{booking.totalAmount.toLocaleString()}</span>
                </div>
              </div>
            </div>
          </div>

          {/* Payment Options */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
            {/* QR Code */}
            <motion.div
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ delay: 0.2 }}
              className="bg-white border-2 border-primary-200 rounded-xl p-6 text-center cursor-pointer hover:border-primary-400 transition-all"
              onClick={handleQRClick}
            >
              <div className="bg-primary-50 rounded-full w-20 h-20 flex items-center justify-center mx-auto mb-4">
                <QrCode className="w-10 h-10 text-primary-600" />
              </div>
              <h3 className="text-lg font-semibold mb-4">Scan QR Code</h3>
              <div className="bg-white p-4 rounded-lg inline-block border-2 border-gray-200">
                <img
                  src={generateQRCodeUrl(booking.totalAmount)}
                  alt="UPI QR Code"
                  className="w-48 h-48 mx-auto"
                />
              </div>
              <p className="text-sm text-gray-600 mt-4">Click to open UPI app</p>
            </motion.div>

            {/* UPI ID */}
            <motion.div
              initial={{ opacity: 0, scale: 0.9 }}
              animate={{ opacity: 1, scale: 1 }}
              transition={{ delay: 0.3 }}
              className="bg-white border-2 border-primary-200 rounded-xl p-6 text-center cursor-pointer hover:border-primary-400 transition-all"
              onClick={handleUPIClick}
            >
              <div className="bg-primary-50 rounded-full w-20 h-20 flex items-center justify-center mx-auto mb-4">
                <Smartphone className="w-10 h-10 text-primary-600" />
              </div>
              <h3 className="text-lg font-semibold mb-4">Pay via UPI ID</h3>
              <div className="bg-gray-50 rounded-lg p-4 mb-4">
                <div className="text-sm text-gray-600 mb-2">UPI ID</div>
                <div className="flex items-center justify-center space-x-2">
                  <span className="text-2xl font-bold text-primary-600">{UPI_ID}</span>
                  <button
                    onClick={(e) => {
                      e.stopPropagation();
                      copyUPIId();
                    }}
                    className="p-2 hover:bg-gray-200 rounded-lg transition-colors"
                    title="Copy UPI ID"
                  >
                    {copied ? (
                      <Check className="w-5 h-5 text-green-600" />
                    ) : (
                      <Copy className="w-5 h-5 text-gray-600" />
                    )}
                  </button>
                </div>
              </div>
              <p className="text-sm text-gray-600">Click to open UPI app</p>
            </motion.div>
          </div>

          {/* Instructions */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4 }}
            className="bg-blue-50 border border-blue-200 rounded-xl p-6"
          >
            <div className="flex items-start">
              <AlertCircle className="w-6 h-6 text-blue-600 mr-3 flex-shrink-0 mt-0.5" />
              <div>
                <h3 className="font-semibold text-blue-900 mb-2">Payment Instructions</h3>
                <ol className="list-decimal list-inside space-y-2 text-blue-800 text-sm">
                  <li>Scan the QR code or click on the UPI ID to open your UPI app</li>
                  <li>Verify the amount and payee name before confirming payment</li>
                  <li>Complete the payment in your UPI app</li>
                  <li>After payment, please contact us with your booking ID: <strong>{booking._id}</strong></li>
                  <li>We will verify your payment and confirm your booking</li>
                </ol>
              </div>
            </div>
          </motion.div>

          {/* Important Note */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.5 }}
            className="mt-6 text-center"
          >
            <p className="text-sm text-gray-600">
              <strong>Note:</strong> Please save your booking ID for reference. After payment, send a screenshot of the payment confirmation to{' '}
              <a href="mailto:balajitourandtravels2026@gmail.com" className="text-primary-600 hover:underline">
                balajitourandtravels2026@gmail.com
              </a>{' '}
              or call us at{' '}
              <a href="tel:9729849875" className="text-primary-600 hover:underline">
                +91 9729849875
              </a>
            </p>
          </motion.div>

          {/* Action Buttons */}
          <div className="flex flex-col sm:flex-row gap-4 mt-8">
            <Link
              to={`/booking/confirmation/${booking._id}`}
              className="btn-secondary flex-1 text-center"
            >
              View Booking Details
            </Link>
            <Link
              to="/packages"
              className="btn-primary flex-1 text-center"
            >
              Browse More Packages
            </Link>
          </div>
        </motion.div>
      </div>
    </div>
  );
};

export default UPIPayment;

