import { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { motion } from 'framer-motion';
import {
  Package,
  BookOpen,
  DollarSign,
  Users,
  Plus,
  Edit,
  Trash2,
  LogOut,
  Eye,
  MessageSquare,
  Mail,
  X,
} from 'lucide-react';
import {
  getPackages,
  getBookings,
  createPackage,
  updatePackage,
  deletePackage,
  updateBookingStatus,
  getMessages,
  updateMessageStatus,
  deleteMessage,
  getMessageStats,
} from '../../services/api';
import toast from 'react-hot-toast';

const AdminDashboard = () => {
  const navigate = useNavigate();
  const [activeTab, setActiveTab] = useState('packages');
  const [packages, setPackages] = useState([]);
  const [bookings, setBookings] = useState([]);
  const [messages, setMessages] = useState([]);
  const [messageStats, setMessageStats] = useState({ total: 0, new: 0 });
  const [loading, setLoading] = useState(true);
  const [showPackageForm, setShowPackageForm] = useState(false);
  const [editingPackage, setEditingPackage] = useState(null);
  const [selectedMessage, setSelectedMessage] = useState(null);
  const [formData, setFormData] = useState({
    title: '',
    slug: '',
    description: '',
    shortDescription: '',
    price: '',
    originalPrice: '',
    duration: '',
    location: '',
    category: 'Domestic',
    images: [''],
    included: [''],
    excluded: [''],
    highlights: [''],
  });

  useEffect(() => {
    if (!localStorage.getItem('adminToken')) {
      navigate('/admin/login');
      return;
    }
    fetchData();
  }, [activeTab, navigate]);

  const fetchData = async () => {
    try {
      setLoading(true);
      if (activeTab === 'packages') {
        const { data } = await getPackages({ limit: 100 });
        setPackages(data.packages || []);
      } else if (activeTab === 'bookings') {
        const { data } = await getBookings({ limit: 100 });
        setBookings(data.bookings || []);
      } else if (activeTab === 'messages') {
        const { data } = await getMessages({ limit: 100 });
        setMessages(data.messages || []);
      }
      
      // Always fetch message stats
      try {
        const { data: statsData } = await getMessageStats();
        setMessageStats(statsData);
      } catch (error) {
        console.error('Failed to load message stats');
      }
    } catch (error) {
      toast.error('Failed to load data');
    } finally {
      setLoading(false);
    }
  };

  const handleLogout = () => {
    localStorage.removeItem('adminToken');
    navigate('/admin/login');
    toast.success('Logged out successfully');
  };

  const handlePackageSubmit = async (e) => {
    e.preventDefault();
    try {
      const packageData = {
        ...formData,
        price: Number(formData.price),
        originalPrice: formData.originalPrice ? Number(formData.originalPrice) : undefined,
        duration: Number(formData.duration),
        images: formData.images.filter((img) => img.trim()),
        included: formData.included.filter((item) => item.trim()),
        excluded: formData.excluded.filter((item) => item.trim()),
        highlights: formData.highlights.filter((item) => item.trim()),
      };

      if (editingPackage) {
        await updatePackage(editingPackage._id, packageData);
        toast.success('Package updated successfully');
      } else {
        await createPackage(packageData);
        toast.success('Package created successfully');
      }

      setShowPackageForm(false);
      setEditingPackage(null);
      resetForm();
      fetchData();
    } catch (error) {
      toast.error(error.response?.data?.message || 'Failed to save package');
    }
  };

  const handleDeletePackage = async (id) => {
    if (!window.confirm('Are you sure you want to delete this package?')) return;

    try {
      await deletePackage(id);
      toast.success('Package deleted successfully');
      fetchData();
    } catch (error) {
      toast.error('Failed to delete package');
    }
  };

  const handleUpdateBookingStatus = async (id, status) => {
    try {
      await updateBookingStatus(id, { status });
      toast.success('Booking status updated');
      fetchData();
    } catch (error) {
      toast.error('Failed to update booking');
    }
  };

  const handleUpdateMessageStatus = async (id, status) => {
    try {
      await updateMessageStatus(id, { status });
      toast.success('Message status updated');
      fetchData();
    } catch (error) {
      toast.error('Failed to update message');
    }
  };

  const handleDeleteMessage = async (id) => {
    if (!window.confirm('Are you sure you want to delete this message?')) return;

    try {
      await deleteMessage(id);
      toast.success('Message deleted successfully');
      fetchData();
    } catch (error) {
      toast.error('Failed to delete message');
    }
  };

  const resetForm = () => {
    setFormData({
      title: '',
      slug: '',
      description: '',
      shortDescription: '',
      price: '',
      originalPrice: '',
      duration: '',
      location: '',
      category: 'Domestic',
      images: [''],
      included: [''],
      excluded: [''],
      highlights: [''],
    });
  };

  useEffect(() => {
    const fetchMessageStats = async () => {
      try {
        const { data } = await getMessageStats();
        setMessageStats(data);
      } catch (error) {
        console.error('Failed to load message stats');
      }
    };
    if (localStorage.getItem('adminToken')) {
      fetchMessageStats();
    }
  }, []);

  const stats = [
    { label: 'Total Packages', value: packages.length, icon: Package, color: 'text-blue-600' },
    { label: 'Total Bookings', value: bookings.length, icon: BookOpen, color: 'text-green-600' },
    {
      label: 'Total Revenue',
      value: `₹${bookings
        .filter((b) => b.paymentStatus === 'paid')
        .reduce((sum, b) => sum + b.totalAmount, 0)
        .toLocaleString()}`,
      icon: DollarSign,
      color: 'text-yellow-600',
    },
    {
      label: 'New Messages',
      value: messageStats.new || 0,
      icon: MessageSquare,
      color: 'text-purple-600',
    },
  ];

  return (
    <div className="min-h-screen bg-gray-50 pt-20">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <h1 className="text-3xl font-bold">Admin Dashboard</h1>
          <button onClick={handleLogout} className="btn-secondary flex items-center">
            <LogOut className="w-5 h-5 mr-2" />
            Logout
          </button>
        </div>

        {/* Stats */}
        <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
          {stats.map((stat, index) => {
            const Icon = stat.icon;
            return (
              <motion.div
                key={index}
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.1 }}
                className="bg-white rounded-xl shadow-md p-6"
              >
                <div className="flex items-center justify-between">
                  <div>
                    <p className="text-gray-600 text-sm mb-1">{stat.label}</p>
                    <p className="text-2xl font-bold">{stat.value}</p>
                  </div>
                  <Icon className={`w-10 h-10 ${stat.color}`} />
                </div>
              </motion.div>
            );
          })}
        </div>

        {/* Tabs */}
        <div className="bg-white rounded-xl shadow-md mb-6">
          <div className="flex border-b">
            <button
              onClick={() => setActiveTab('packages')}
              className={`px-6 py-4 font-semibold ${
                activeTab === 'packages'
                  ? 'text-primary-600 border-b-2 border-primary-600'
                  : 'text-gray-600 hover:text-primary-600'
              }`}
            >
              Packages
            </button>
            <button
              onClick={() => setActiveTab('bookings')}
              className={`px-6 py-4 font-semibold ${
                activeTab === 'bookings'
                  ? 'text-primary-600 border-b-2 border-primary-600'
                  : 'text-gray-600 hover:text-primary-600'
              }`}
            >
              Bookings
            </button>
            <button
              onClick={() => setActiveTab('messages')}
              className={`px-6 py-4 font-semibold ${
                activeTab === 'messages'
                  ? 'text-primary-600 border-b-2 border-primary-600'
                  : 'text-gray-600 hover:text-primary-600'
              }`}
            >
              Messages
            </button>
          </div>
        </div>

        {/* Content */}
        {activeTab === 'packages' ? (
          <div>
            <div className="flex justify-between items-center mb-6">
              <h2 className="text-2xl font-bold">Tour Packages</h2>
              <button
                onClick={() => {
                  setEditingPackage(null);
                  resetForm();
                  setShowPackageForm(true);
                }}
                className="btn-primary flex items-center"
              >
                <Plus className="w-5 h-5 mr-2" />
                Add Package
              </button>
            </div>

            {loading ? (
              <div className="text-center py-12">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary-600 mx-auto"></div>
              </div>
            ) : (
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                {packages.map((pkg) => (
                  <div key={pkg._id} className="bg-white rounded-xl shadow-md overflow-hidden">
                    <img
                      src={pkg.images[0]}
                      alt={pkg.title}
                      className="w-full h-48 object-cover"
                    />
                    <div className="p-4">
                      <h3 className="font-bold mb-2">{pkg.title}</h3>
                      <p className="text-sm text-gray-600 mb-2">{pkg.location}</p>
                      <p className="font-semibold text-primary-600 mb-4">
                        ₹{pkg.price.toLocaleString()}
                      </p>
                      <div className="flex gap-2">
                        <button
                          onClick={() => {
                            setEditingPackage(pkg);
                            setFormData({
                              ...pkg,
                              images: pkg.images.length ? pkg.images : [''],
                              included: pkg.included.length ? pkg.included : [''],
                              excluded: pkg.excluded.length ? pkg.excluded : [''],
                              highlights: pkg.highlights.length ? pkg.highlights : [''],
                            });
                            setShowPackageForm(true);
                          }}
                          className="flex-1 btn-secondary text-sm py-2 flex items-center justify-center"
                        >
                          <Edit className="w-4 h-4 mr-1" />
                          Edit
                        </button>
                        <button
                          onClick={() => handleDeletePackage(pkg._id)}
                          className="flex-1 bg-red-600 text-white px-4 py-2 rounded-lg hover:bg-red-700 text-sm flex items-center justify-center"
                        >
                          <Trash2 className="w-4 h-4 mr-1" />
                          Delete
                        </button>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </div>
        ) : activeTab === 'bookings' ? (
          <div>
            <h2 className="text-2xl font-bold mb-6">Bookings</h2>
            {loading ? (
              <div className="text-center py-12">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary-600 mx-auto"></div>
              </div>
            ) : (
              <div className="bg-white rounded-xl shadow-md overflow-hidden">
                <table className="w-full">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Booking ID
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Package
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Customer
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Amount
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Payment
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Status
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Actions
                      </th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-200">
                    {bookings.map((booking) => (
                      <tr key={booking._id}>
                        <td className="px-6 py-4 text-sm font-mono">
                          {booking._id.slice(-8)}
                        </td>
                        <td className="px-6 py-4 text-sm">{booking.packageName}</td>
                        <td className="px-6 py-4 text-sm">
                          <div>{booking.name}</div>
                          <div className="text-gray-500 text-xs">{booking.email}</div>
                        </td>
                        <td className="px-6 py-4 text-sm font-semibold">
                          ₹{booking.totalAmount.toLocaleString()}
                        </td>
                        <td className="px-6 py-4">
                          <span
                            className={`px-2 py-1 rounded-full text-xs font-semibold ${
                              booking.paymentStatus === 'paid'
                                ? 'bg-green-100 text-green-800'
                                : booking.paymentStatus === 'pending'
                                ? 'bg-yellow-100 text-yellow-800'
                                : 'bg-red-100 text-red-800'
                            }`}
                          >
                            {booking.paymentStatus}
                          </span>
                        </td>
                        <td className="px-6 py-4">
                          <select
                            value={booking.status}
                            onChange={(e) => handleUpdateBookingStatus(booking._id, e.target.value)}
                            className="text-sm border border-gray-300 rounded px-2 py-1"
                          >
                            <option value="pending">Pending</option>
                            <option value="confirmed">Confirmed</option>
                            <option value="cancelled">Cancelled</option>
                            <option value="completed">Completed</option>
                          </select>
                        </td>
                        <td className="px-6 py-4">
                          <a
                            href={`/booking/confirmation/${booking._id}`}
                            target="_blank"
                            className="text-primary-600 hover:underline text-sm"
                          >
                            View
                          </a>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>
        ) : (
          <div>
            <h2 className="text-2xl font-bold mb-6">Messages</h2>
            {loading ? (
              <div className="text-center py-12">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-primary-600 mx-auto"></div>
              </div>
            ) : messages.length === 0 ? (
              <div className="text-center py-12 bg-white rounded-xl shadow-md">
                <MessageSquare className="w-16 h-16 text-gray-400 mx-auto mb-4" />
                <p className="text-gray-600">No messages yet</p>
              </div>
            ) : (
              <div className="bg-white rounded-xl shadow-md overflow-hidden">
                <table className="w-full">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Date
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Name
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Email
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Phone
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Subject
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Status
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">
                        Actions
                      </th>
                    </tr>
                  </thead>
                  <tbody className="divide-y divide-gray-200">
                    {messages.map((msg) => (
                      <tr key={msg._id} className={msg.status === 'new' ? 'bg-blue-50' : ''}>
                        <td className="px-6 py-4 text-sm">
                          {new Date(msg.createdAt).toLocaleDateString()}
                        </td>
                        <td className="px-6 py-4 text-sm font-medium">{msg.name}</td>
                        <td className="px-6 py-4 text-sm">
                          <a
                            href={`mailto:${msg.email}`}
                            className="text-primary-600 hover:underline"
                          >
                            {msg.email}
                          </a>
                        </td>
                        <td className="px-6 py-4 text-sm">
                          <a
                            href={`tel:${msg.phone}`}
                            className="text-primary-600 hover:underline"
                          >
                            {msg.phone}
                          </a>
                        </td>
                        <td className="px-6 py-4 text-sm">{msg.subject}</td>
                        <td className="px-6 py-4">
                          <select
                            value={msg.status}
                            onChange={(e) => handleUpdateMessageStatus(msg._id, e.target.value)}
                            className="text-sm border border-gray-300 rounded px-2 py-1"
                          >
                            <option value="new">New</option>
                            <option value="read">Read</option>
                            <option value="replied">Replied</option>
                            <option value="archived">Archived</option>
                          </select>
                        </td>
                        <td className="px-6 py-4">
                          <div className="flex gap-2">
                            <button
                              onClick={() => {
                                setSelectedMessage(msg);
                                // Mark as read if it's new
                                if (msg.status === 'new') {
                                  handleUpdateMessageStatus(msg._id, 'read');
                                }
                              }}
                              className="text-primary-600 hover:underline text-sm flex items-center"
                            >
                              <Eye className="w-4 h-4 mr-1" />
                              View
                            </button>
                            <button
                              onClick={() => handleDeleteMessage(msg._id)}
                              className="text-red-600 hover:underline text-sm flex items-center"
                            >
                              <Trash2 className="w-4 h-4 mr-1" />
                              Delete
                            </button>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>
        )}
      </div>

      {/* Package Form Modal */}
      {showPackageForm && (
        <div className="fixed inset-0 bg-black/50 z-50 flex items-center justify-center p-4">
          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            className="bg-white rounded-xl shadow-2xl p-8 max-w-2xl w-full max-h-[90vh] overflow-y-auto"
          >
            <h2 className="text-2xl font-bold mb-6">
              {editingPackage ? 'Edit Package' : 'Add New Package'}
            </h2>
            <form onSubmit={handlePackageSubmit} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-2">Title *</label>
                  <input
                    type="text"
                    value={formData.title}
                    onChange={(e) => setFormData({ ...formData, title: e.target.value })}
                    required
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-2">Slug *</label>
                  <input
                    type="text"
                    value={formData.slug}
                    onChange={(e) =>
                      setFormData({ ...formData, slug: e.target.value.toLowerCase().replace(/\s+/g, '-') })
                    }
                    required
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                  />
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Short Description *</label>
                <input
                  type="text"
                  value={formData.shortDescription}
                  onChange={(e) => setFormData({ ...formData, shortDescription: e.target.value })}
                  required
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Description *</label>
                <textarea
                  value={formData.description}
                  onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                  required
                  rows="4"
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                />
              </div>
              <div className="grid grid-cols-4 gap-4">
                <div>
                  <label className="block text-sm font-medium mb-2">Price *</label>
                  <input
                    type="number"
                    value={formData.price}
                    onChange={(e) => setFormData({ ...formData, price: e.target.value })}
                    required
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-2">Original Price</label>
                  <input
                    type="number"
                    value={formData.originalPrice}
                    onChange={(e) => setFormData({ ...formData, originalPrice: e.target.value })}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-2">Duration *</label>
                  <input
                    type="number"
                    value={formData.duration}
                    onChange={(e) => setFormData({ ...formData, duration: e.target.value })}
                    required
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium mb-2">Category *</label>
                  <select
                    value={formData.category}
                    onChange={(e) => setFormData({ ...formData, category: e.target.value })}
                    className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                  >
                    <option>Domestic</option>
                    <option>International</option>
                    <option>Adventure</option>
                    <option>Honeymoon</option>
                    <option>Family</option>
                    <option>Luxury</option>
                  </select>
                </div>
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Location *</label>
                <input
                  type="text"
                  value={formData.location}
                  onChange={(e) => setFormData({ ...formData, location: e.target.value })}
                  required
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                />
              </div>
              <div>
                <label className="block text-sm font-medium mb-2">Images (URLs, one per line)</label>
                <textarea
                  value={formData.images.join('\n')}
                  onChange={(e) =>
                    setFormData({ ...formData, images: e.target.value.split('\n') })
                  }
                  rows="3"
                  className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                />
              </div>
              <div className="flex gap-4">
                <button type="submit" className="btn-primary flex-1">
                  {editingPackage ? 'Update' : 'Create'} Package
                </button>
                <button
                  type="button"
                  onClick={() => {
                    setShowPackageForm(false);
                    setEditingPackage(null);
                    resetForm();
                  }}
                  className="btn-secondary flex-1"
                >
                  Cancel
                </button>
              </div>
            </form>
          </motion.div>
        </div>
      )}

      {/* Message Detail Modal */}
      {selectedMessage && (
        <div className="fixed inset-0 bg-black/50 z-50 flex items-center justify-center p-4">
          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            className="bg-white rounded-xl shadow-2xl p-8 max-w-2xl w-full max-h-[90vh] overflow-y-auto"
          >
            <div className="flex items-center justify-between mb-6">
              <h2 className="text-2xl font-bold">Message Details</h2>
              <button
                onClick={() => setSelectedMessage(null)}
                className="text-gray-400 hover:text-gray-600"
              >
                <X className="w-6 h-6" />
              </button>
            </div>

            <div className="space-y-4">
              <div>
                <label className="text-sm font-medium text-gray-600">Date</label>
                <p className="text-gray-900">
                  {new Date(selectedMessage.createdAt).toLocaleString()}
                </p>
              </div>

              <div>
                <label className="text-sm font-medium text-gray-600">Name</label>
                <p className="text-gray-900">{selectedMessage.name}</p>
              </div>

              <div>
                <label className="text-sm font-medium text-gray-600">Email</label>
                <p className="text-gray-900">
                  <a
                    href={`mailto:${selectedMessage.email}`}
                    className="text-primary-600 hover:underline"
                  >
                    {selectedMessage.email}
                  </a>
                </p>
              </div>

              <div>
                <label className="text-sm font-medium text-gray-600">Phone</label>
                <p className="text-gray-900">
                  <a
                    href={`tel:${selectedMessage.phone}`}
                    className="text-primary-600 hover:underline"
                  >
                    {selectedMessage.phone}
                  </a>
                </p>
              </div>

              <div>
                <label className="text-sm font-medium text-gray-600">Subject</label>
                <p className="text-gray-900">{selectedMessage.subject}</p>
              </div>

              <div>
                <label className="text-sm font-medium text-gray-600">Message</label>
                <p className="text-gray-900 whitespace-pre-wrap bg-gray-50 p-4 rounded-lg">
                  {selectedMessage.message}
                </p>
              </div>

              <div>
                <label className="text-sm font-medium text-gray-600">Status</label>
                <select
                  value={selectedMessage.status}
                  onChange={(e) => {
                    handleUpdateMessageStatus(selectedMessage._id, e.target.value);
                    setSelectedMessage({ ...selectedMessage, status: e.target.value });
                  }}
                  className="mt-1 text-sm border border-gray-300 rounded px-3 py-2"
                >
                  <option value="new">New</option>
                  <option value="read">Read</option>
                  <option value="replied">Replied</option>
                  <option value="archived">Archived</option>
                </select>
              </div>
            </div>

            <div className="flex gap-4 mt-6">
              <a
                href={`mailto:${selectedMessage.email}?subject=Re: ${selectedMessage.subject}`}
                className="btn-primary flex items-center justify-center flex-1"
              >
                <Mail className="w-5 h-5 mr-2" />
                Reply via Email
              </a>
              <button
                onClick={() => {
                  handleDeleteMessage(selectedMessage._id);
                  setSelectedMessage(null);
                }}
                className="bg-red-600 text-white px-6 py-3 rounded-lg hover:bg-red-700 flex items-center"
              >
                <Trash2 className="w-5 h-5 mr-2" />
                Delete
              </button>
            </div>
          </motion.div>
        </div>
      )}
    </div>
  );
};

export default AdminDashboard;

