import axios from 'axios';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:5000/api';

const api = axios.create({
  baseURL: API_URL,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Add token to requests if available
api.interceptors.request.use((config) => {
  const token = localStorage.getItem('adminToken');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

// Response interceptor for error handling
api.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.code === 'ECONNREFUSED' || error.message === 'Network Error') {
      console.error('❌ Cannot connect to server. Make sure the server is running on port 5000');
      error.message = 'Cannot connect to server. Please check if the server is running.';
    }
    return Promise.reject(error);
  }
);

// Packages API
export const getPackages = (params) => api.get('/packages', { params });
export const getPackageBySlug = (slug) => api.get(`/packages/slug/${slug}`);
export const getFeaturedPackages = () => api.get('/packages/featured');

// Bookings API
export const createBooking = (data) => api.post('/bookings', data);
export const getBookingById = (id) => api.get(`/bookings/${id}`);

// Payments API
export const createPayment = (data) => api.post('/payments/create', data);
export const verifyPayment = (bookingId) => api.get(`/payments/verify/${bookingId}`);

// Auth API
export const adminLogin = (data) => api.post('/auth/login', data);
export const getMe = () => api.get('/auth/me');

// Admin APIs
export const createPackage = (data) => api.post('/packages', data);
export const updatePackage = (id, data) => api.put(`/packages/${id}`, data);
export const deletePackage = (id) => api.delete(`/packages/${id}`);
export const getBookings = (params) => api.get('/bookings', { params });
export const updateBookingStatus = (id, data) => api.put(`/bookings/${id}/status`, data);

// Messages API
export const createMessage = (data) => api.post('/messages', data);
export const getMessages = (params) => api.get('/messages', { params });
export const getMessageById = (id) => api.get(`/messages/${id}`);
export const updateMessageStatus = (id, data) => api.put(`/messages/${id}/status`, data);
export const deleteMessage = (id) => api.delete(`/messages/${id}`);
export const getMessageStats = () => api.get('/messages/stats');

export default api;

