#!/bin/bash

# Comprehensive script to fix all issues in the project

echo "🔧 Fixing All Issues..."
echo "======================"
echo ""

# Colors for output
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Function to check if command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# 1. Create server/.env if it doesn't exist
echo "1️⃣  Checking server/.env..."
if [ ! -f "server/.env" ]; then
    echo "   Creating server/.env..."
    cat > server/.env << 'EOF'
# Server Configuration
PORT=5000
NODE_ENV=development

# MongoDB - Update this with your MongoDB connection string
# For local MongoDB (if installed):
MONGODB_URI=mongodb://localhost:27017/balaji-tour-travel

# For MongoDB Atlas (cloud - recommended):
# MONGODB_URI=mongodb+srv://username:password@cluster0.xxxxx.mongodb.net/balaji-tour-travel?retryWrites=true&w=majority

# JWT Secret - Change this in production!
JWT_SECRET=balaji-tour-travel-super-secret-jwt-key-2024-min-32-chars

# Admin Default Credentials
ADMIN_EMAIL=admin@balajitour.com
ADMIN_PASSWORD=admin123

# Veegah Payment Gateway (Update with your credentials)
VEEGAH_MERCHANT_ID=your_merchant_id
VEEGAH_API_KEY=your_api_key
VEEGAH_SECRET_KEY=your_secret_key
VEEGAH_PAYMENT_URL=https://api.veegah.com/payment
VEEGAH_CALLBACK_URL=http://localhost:5000/api/payments/callback

# Frontend URL
CLIENT_URL=http://localhost:5173
EOF
    echo -e "   ${GREEN}✅ Created server/.env${NC}"
else
    echo -e "   ${YELLOW}ℹ️  server/.env already exists${NC}"
fi

# 2. Create client/.env if it doesn't exist
echo ""
echo "2️⃣  Checking client/.env..."
if [ ! -f "client/.env" ]; then
    echo "   Creating client/.env..."
    echo "VITE_API_URL=http://localhost:5000/api" > client/.env
    echo -e "   ${GREEN}✅ Created client/.env${NC}"
else
    echo -e "   ${YELLOW}ℹ️  client/.env already exists${NC}"
fi

# 3. Check MongoDB
echo ""
echo "3️⃣  Checking MongoDB..."
if pgrep mongod > /dev/null; then
    echo -e "   ${GREEN}✅ MongoDB is running${NC}"
elif command_exists mongod; then
    echo -e "   ${YELLOW}⚠️  MongoDB is installed but not running${NC}"
    echo "   Start it with: sudo systemctl start mongod"
else
    echo -e "   ${RED}❌ MongoDB is not installed${NC}"
    echo ""
    echo "   📖 MongoDB Setup Options:"
    echo "   Option 1: Use MongoDB Atlas (Cloud - Recommended)"
    echo "            → Sign up: https://www.mongodb.com/cloud/atlas/register"
    echo "            → Create free cluster"
    echo "            → Update MONGODB_URI in server/.env"
    echo ""
    echo "   Option 2: Install MongoDB locally"
    echo "            → See MONGODB_SETUP.md for instructions"
fi

# 4. Check port 5000
echo ""
echo "4️⃣  Checking port 5000..."
if lsof -ti :5000 > /dev/null 2>&1; then
    PID=$(lsof -ti :5000)
    echo -e "   ${YELLOW}⚠️  Port 5000 is in use by PID: $PID${NC}"
    echo "   Kill it with: sudo kill -9 $PID"
    echo "   Or use: ./kill-port.sh"
else
    echo -e "   ${GREEN}✅ Port 5000 is free${NC}"
fi

# 5. Check dependencies
echo ""
echo "5️⃣  Checking dependencies..."
if [ -d "server/node_modules" ]; then
    echo -e "   ${GREEN}✅ Server dependencies installed${NC}"
else
    echo -e "   ${RED}❌ Server dependencies missing${NC}"
    echo "   Installing server dependencies..."
    cd server && npm install && cd ..
    echo -e "   ${GREEN}✅ Server dependencies installed${NC}"
fi

if [ -d "client/node_modules" ]; then
    echo -e "   ${GREEN}✅ Client dependencies installed${NC}"
else
    echo -e "   ${RED}❌ Client dependencies missing${NC}"
    echo "   Installing client dependencies..."
    cd client && npm install && cd ..
    echo -e "   ${GREEN}✅ Client dependencies installed${NC}"
fi

# 6. Summary
echo ""
echo "======================"
echo "📋 Summary:"
echo "======================"
echo ""
echo "✅ Environment files created/checked"
echo "✅ Dependencies checked"
echo ""
echo "📝 Next Steps:"
echo ""
echo "1. Set up MongoDB:"
echo "   - Option A: Use MongoDB Atlas (recommended)"
echo "     → https://www.mongodb.com/cloud/atlas/register"
echo "     → Update MONGODB_URI in server/.env"
echo ""
echo "   - Option B: Install MongoDB locally"
echo "     → See MONGODB_SETUP.md"
echo ""
echo "2. After MongoDB is ready:"
echo "   cd server"
echo "   npm run seed    # Seed database with 10 packages"
echo "   npm run dev     # Start server"
echo ""
echo "3. In another terminal:"
echo "   cd client"
echo "   npm run dev     # Start frontend"
echo ""
echo "4. Open browser:"
echo "   http://localhost:5173"
echo ""
echo "📖 For detailed instructions, see:"
echo "   - MONGODB_SETUP.md (MongoDB installation)"
echo "   - MONGODB_QUICK_FIX.md (Quick MongoDB fixes)"
echo "   - TROUBLESHOOTING.md (Troubleshooting guide)"
echo ""

