import Booking from '../models/Booking.js';
import Package from '../models/Package.js';

export const createBooking = async (req, res) => {
  try {
    const { packageId, name, email, phone, travelDate, numberOfTravelers, specialRequests } = req.body;

    const packageData = await Package.findById(packageId);
    if (!packageData || !packageData.isActive) {
      return res.status(404).json({ message: 'Package not found' });
    }

    const totalAmount = packageData.price * numberOfTravelers;

    const booking = new Booking({
      packageId,
      packageName: packageData.title,
      name,
      email,
      phone,
      travelDate: new Date(travelDate),
      numberOfTravelers,
      totalAmount,
      specialRequests,
      paymentStatus: 'pending',
      status: 'pending',
    });

    await booking.save();

    res.status(201).json({
      booking,
      message: 'Booking created successfully. Proceed to payment.',
    });
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

export const getBookings = async (req, res) => {
  try {
    const { page = 1, limit = 20, status, paymentStatus } = req.query;
    const query = {};

    if (status) query.status = status;
    if (paymentStatus) query.paymentStatus = paymentStatus;

    const bookings = await Booking.find(query)
      .populate('packageId', 'title images location')
      .sort({ createdAt: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit);

    const total = await Booking.countDocuments(query);

    res.json({
      bookings,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total,
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const getBookingById = async (req, res) => {
  try {
    const booking = await Booking.findById(req.params.id).populate('packageId');
    if (!booking) {
      return res.status(404).json({ message: 'Booking not found' });
    }
    res.json(booking);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const updateBookingStatus = async (req, res) => {
  try {
    const { status, paymentStatus } = req.body;
    const booking = await Booking.findByIdAndUpdate(
      req.params.id,
      { status, paymentStatus },
      { new: true, runValidators: true }
    );
    if (!booking) {
      return res.status(404).json({ message: 'Booking not found' });
    }
    res.json(booking);
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

export const updatePaymentStatus = async (req, res) => {
  try {
    const { bookingId, paymentStatus, veegahTransactionId, paymentId } = req.body;

    const booking = await Booking.findByIdAndUpdate(
      bookingId,
      {
        paymentStatus,
        veegahTransactionId,
        paymentId,
        status: paymentStatus === 'paid' ? 'confirmed' : booking.status,
      },
      { new: true }
    );

    if (!booking) {
      return res.status(404).json({ message: 'Booking not found' });
    }

    res.json(booking);
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

