import Message from '../models/Message.js';

export const createMessage = async (req, res) => {
  try {
    const { name, email, phone, subject, message } = req.body;

    const newMessage = new Message({
      name,
      email,
      phone,
      subject,
      message,
      status: 'new',
    });

    await newMessage.save();

    res.status(201).json({
      message: newMessage,
      success: true,
    });
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

export const getMessages = async (req, res) => {
  try {
    const { page = 1, limit = 20, status } = req.query;
    const query = {};

    if (status) query.status = status;

    const messages = await Message.find(query)
      .sort({ createdAt: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit);

    const total = await Message.countDocuments(query);

    res.json({
      messages,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total,
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const getMessageById = async (req, res) => {
  try {
    const message = await Message.findById(req.params.id);
    if (!message) {
      return res.status(404).json({ message: 'Message not found' });
    }
    res.json(message);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const updateMessageStatus = async (req, res) => {
  try {
    const { status, adminNotes } = req.body;
    const updateData = { status };
    
    if (status === 'replied') {
      updateData.repliedAt = new Date();
    }
    
    if (adminNotes) {
      updateData.adminNotes = adminNotes;
    }

    const message = await Message.findByIdAndUpdate(
      req.params.id,
      updateData,
      { new: true, runValidators: true }
    );

    if (!message) {
      return res.status(404).json({ message: 'Message not found' });
    }

    res.json(message);
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

export const deleteMessage = async (req, res) => {
  try {
    const message = await Message.findByIdAndDelete(req.params.id);
    if (!message) {
      return res.status(404).json({ message: 'Message not found' });
    }
    res.json({ message: 'Message deleted successfully' });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const getMessageStats = async (req, res) => {
  try {
    const total = await Message.countDocuments();
    const newMessages = await Message.countDocuments({ status: 'new' });
    const readMessages = await Message.countDocuments({ status: 'read' });
    const repliedMessages = await Message.countDocuments({ status: 'replied' });

    res.json({
      total,
      new: newMessages,
      read: readMessages,
      replied: repliedMessages,
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

