import Package from '../models/Package.js';

export const getPackages = async (req, res) => {
  try {
    const {
      page = 1,
      limit = 12,
      category,
      location,
      minPrice,
      maxPrice,
      minDuration,
      maxDuration,
      search,
      featured,
    } = req.query;

    const query = { isActive: true };

    if (category) query.category = category;
    if (location) query.location = { $regex: location, $options: 'i' };
    if (minPrice || maxPrice) {
      query.price = {};
      if (minPrice) query.price.$gte = Number(minPrice);
      if (maxPrice) query.price.$lte = Number(maxPrice);
    }
    if (minDuration || maxDuration) {
      query.duration = {};
      if (minDuration) query.duration.$gte = Number(minDuration);
      if (maxDuration) query.duration.$lte = Number(maxDuration);
    }
    if (search) {
      query.$or = [
        { title: { $regex: search, $options: 'i' } },
        { location: { $regex: search, $options: 'i' } },
        { description: { $regex: search, $options: 'i' } },
      ];
    }
    if (featured === 'true') query.featured = true;

    const packages = await Package.find(query)
      .sort({ featured: -1, createdAt: -1 })
      .limit(limit * 1)
      .skip((page - 1) * limit);

    const total = await Package.countDocuments(query);

    res.json({
      packages,
      totalPages: Math.ceil(total / limit),
      currentPage: page,
      total,
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const getPackageById = async (req, res) => {
  try {
    const tourPackage = await Package.findById(req.params.id);
    if (!tourPackage || !tourPackage.isActive) {
      return res.status(404).json({ message: 'Package not found' });
    }
    res.json(tourPackage);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const getPackageBySlug = async (req, res) => {
  try {
    const tourPackage = await Package.findOne({ slug: req.params.slug });
    if (!tourPackage || !tourPackage.isActive) {
      return res.status(404).json({ message: 'Package not found' });
    }
    res.json(tourPackage);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const createPackage = async (req, res) => {
  try {
    const tourPackage = new Package(req.body);
    await tourPackage.save();
    res.status(201).json(tourPackage);
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

export const updatePackage = async (req, res) => {
  try {
    const tourPackage = await Package.findByIdAndUpdate(req.params.id, req.body, {
      new: true,
      runValidators: true,
    });
    if (!tourPackage) {
      return res.status(404).json({ message: 'Package not found' });
    }
    res.json(tourPackage);
  } catch (error) {
    res.status(400).json({ message: error.message });
  }
};

export const deletePackage = async (req, res) => {
  try {
    const tourPackage = await Package.findByIdAndDelete(req.params.id);
    if (!tourPackage) {
      return res.status(404).json({ message: 'Package not found' });
    }
    res.json({ message: 'Package deleted successfully' });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

export const getFeaturedPackages = async (req, res) => {
  try {
    const packages = await Package.find({ featured: true, isActive: true })
      .limit(6)
      .sort({ createdAt: -1 });
    res.json(packages);
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

