import axios from 'axios';
import Booking from '../models/Booking.js';
import crypto from 'crypto';

// Generate payment request for Veegah Gateway
export const createPayment = async (req, res) => {
  try {
    const { bookingId } = req.body;

    const booking = await Booking.findById(bookingId);
    if (!booking) {
      return res.status(404).json({ message: 'Booking not found' });
    }

    if (booking.paymentStatus === 'paid') {
      return res.status(400).json({ message: 'Booking already paid' });
    }

    const merchantId = process.env.VEEGAH_MERCHANT_ID;
    const apiKey = process.env.VEEGAH_API_KEY;
    const secretKey = process.env.VEEGAH_SECRET_KEY;
    const callbackUrl = process.env.VEEGAH_CALLBACK_URL || `${process.env.CLIENT_URL}/payment/callback`;

    // Create payment payload
    const paymentData = {
      merchant_id: merchantId,
      amount: booking.totalAmount,
      currency: 'INR',
      order_id: booking._id.toString(),
      customer_name: booking.name,
      customer_email: booking.email,
      customer_phone: booking.phone,
      description: `Payment for ${booking.packageName}`,
      callback_url: callbackUrl,
      return_url: `${process.env.CLIENT_URL}/booking/confirmation/${booking._id}`,
    };

    // Generate signature
    const signatureString = `${merchantId}${paymentData.amount}${paymentData.order_id}${secretKey}`;
    const signature = crypto.createHash('sha256').update(signatureString).digest('hex');
    paymentData.signature = signature;

    // In production, make actual API call to Veegah
    // For now, return payment URL structure
    const paymentUrl = process.env.VEEGAH_PAYMENT_URL || 'https://api.veegah.com/payment';

    res.json({
      paymentUrl: `${paymentUrl}?${new URLSearchParams(paymentData).toString()}`,
      paymentData,
      bookingId: booking._id,
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

// Handle payment callback from Veegah
export const paymentCallback = async (req, res) => {
  try {
    const {
      order_id,
      transaction_id,
      status,
      amount,
      signature,
    } = req.body;

    const booking = await Booking.findById(order_id);
    if (!booking) {
      return res.status(404).json({ message: 'Booking not found' });
    }

    // Verify signature
    const secretKey = process.env.VEEGAH_SECRET_KEY;
    const expectedSignature = crypto
      .createHash('sha256')
      .update(`${process.env.VEEGAH_MERCHANT_ID}${amount}${order_id}${secretKey}`)
      .digest('hex');

    if (signature !== expectedSignature) {
      return res.status(400).json({ message: 'Invalid signature' });
    }

    // Update booking status
    booking.paymentStatus = status === 'success' ? 'paid' : 'failed';
    booking.veegahTransactionId = transaction_id;
    booking.paymentId = transaction_id;
    booking.status = status === 'success' ? 'confirmed' : 'pending';

    await booking.save();

    res.json({
      success: status === 'success',
      booking,
      message: status === 'success' ? 'Payment successful' : 'Payment failed',
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

// Verify payment status
export const verifyPayment = async (req, res) => {
  try {
    const { bookingId } = req.params;

    const booking = await Booking.findById(bookingId);
    if (!booking) {
      return res.status(404).json({ message: 'Booking not found' });
    }

    res.json({
      paymentStatus: booking.paymentStatus,
      bookingStatus: booking.status,
      transactionId: booking.veegahTransactionId,
    });
  } catch (error) {
    res.status(500).json({ message: error.message });
  }
};

