import dotenv from 'dotenv';
import connectDB from '../config/db.js';
import User from '../models/User.js';
import bcrypt from 'bcryptjs';

dotenv.config();

const resetAdmin = async () => {
  try {
    await connectDB();
    
    const adminEmail = process.env.ADMIN_EMAIL || 'admin@balajitour.com';
    const adminPassword = process.env.ADMIN_PASSWORD || 'admin123';
    
    console.log('🔄 Resetting admin user...');
    console.log('Email:', adminEmail);
    console.log('Password:', adminPassword);
    console.log('');
    
    // Delete existing admin user
    const deleted = await User.deleteMany({ email: adminEmail });
    console.log(`🗑️  Deleted ${deleted.deletedCount} existing admin user(s)`);
    
    // Create fresh admin user
    // Pass plain password - pre-save hook will hash it
    const admin = await User.create({
      email: adminEmail,
      password: adminPassword, // Plain password - will be hashed by pre-save hook
      role: 'admin',
      name: 'Admin',
    });
    
    console.log('✅ New admin user created!');
    console.log('ID:', admin._id);
    console.log('Email:', admin.email);
    console.log('Role:', admin.role);
    console.log('');
    
    // Verify password works
    console.log('🧪 Verifying password...');
    const testMatch = await admin.comparePassword(adminPassword);
    
    if (testMatch) {
      console.log('✅ Password verification successful!');
    } else {
      console.log('❌ Password verification failed!');
    }
    
    // List all users
    console.log('');
    console.log('📋 All users in database:');
    const allUsers = await User.find({});
    allUsers.forEach(user => {
      console.log(`  - ${user.email} (${user.role})`);
    });
    
    console.log('');
    console.log('✅ Admin reset complete!');
    console.log('');
    console.log('📋 Login Credentials:');
    console.log('   Email:', adminEmail);
    console.log('   Password:', adminPassword);
    console.log('');
    console.log('⚠️  Please restart your server for changes to take effect!');
    
    process.exit(0);
  } catch (error) {
    console.error('❌ Error:', error.message);
    process.exit(1);
  }
};

resetAdmin();

