import dotenv from 'dotenv';
import connectDB from '../config/db.js';
import User from '../models/User.js';
import bcrypt from 'bcryptjs';

dotenv.config();

const testLogin = async () => {
  try {
    await connectDB();
    
    const adminEmail = process.env.ADMIN_EMAIL || 'admin@balajitour.com';
    const adminPassword = process.env.ADMIN_PASSWORD || 'admin123';
    
    console.log('🧪 Testing login...');
    console.log('Email:', adminEmail);
    console.log('Password:', adminPassword);
    console.log('');
    
    // Find user
    const user = await User.findOne({ email: adminEmail.toLowerCase().trim() });
    
    if (!user) {
      console.log('❌ User not found!');
      process.exit(1);
    }
    
    console.log('✅ User found:');
    console.log('  ID:', user._id);
    console.log('  Email:', user.email);
    console.log('  Role:', user.role);
    console.log('  Password hash:', user.password.substring(0, 20) + '...');
    console.log('');
    
    // Test password comparison
    console.log('🔐 Testing password comparison...');
    
    // Method 1: Using comparePassword method
    const method1 = await user.comparePassword(adminPassword);
    console.log('Method 1 (comparePassword):', method1 ? '✅ Match' : '❌ No match');
    
    // Method 2: Direct bcrypt compare
    const method2 = await bcrypt.compare(adminPassword, user.password);
    console.log('Method 2 (direct bcrypt):', method2 ? '✅ Match' : '❌ No match');
    
    // Test with wrong password
    const wrongMatch = await bcrypt.compare('wrongpassword', user.password);
    console.log('Wrong password test:', wrongMatch ? '❌ Should not match!' : '✅ Correctly rejected');
    
    console.log('');
    
    if (method1 || method2) {
      console.log('✅ Password is correct! Login should work.');
    } else {
      console.log('❌ Password does not match!');
      console.log('');
      console.log('🔧 Fixing password...');
      
      // Delete and recreate
      await User.deleteOne({ _id: user._id });
      
      const hashedPassword = await bcrypt.hash(adminPassword, 12);
      const newUser = await User.create({
        email: adminEmail,
        password: adminPassword, // Let the pre-save hook hash it
        role: 'admin',
        name: 'Admin',
      });
      
      // Wait a bit for save to complete
      await new Promise(resolve => setTimeout(resolve, 100));
      
      // Reload user
      const reloadedUser = await User.findById(newUser._id);
      const testMatch = await reloadedUser.comparePassword(adminPassword);
      
      console.log('New user password test:', testMatch ? '✅ Match' : '❌ No match');
    }
    
    process.exit(0);
  } catch (error) {
    console.error('❌ Error:', error.message);
    console.error(error.stack);
    process.exit(1);
  }
};

testLogin();

