import express from 'express';
import dotenv from 'dotenv';
import cors from 'cors';
import connectDB from './config/db.js';
import User from './models/User.js';
import bcrypt from 'bcryptjs';

// Load env vars
dotenv.config();

// Import routes
import authRoutes from './routes/authRoutes.js';
import packageRoutes from './routes/packageRoutes.js';
import bookingRoutes from './routes/bookingRoutes.js';
import paymentRoutes from './routes/paymentRoutes.js';
import messageRoutes from './routes/messageRoutes.js';

// Connect to database
connectDB();

const app = express();

// Middleware
app.use(cors({
  origin: process.env.CLIENT_URL || 'http://localhost:5173',
  credentials: true,
}));
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// Routes
app.use('/api/auth', authRoutes);
app.use('/api/packages', packageRoutes);
app.use('/api/bookings', bookingRoutes);
app.use('/api/payments', paymentRoutes);
app.use('/api/messages', messageRoutes);

// Health check
app.get('/api/health', (req, res) => {
  res.json({ status: 'OK', message: 'Balaji Tour & Travel API is running' });
});

// Initialize admin user if not exists
const initializeAdmin = async () => {
  try {
    const adminExists = await User.findOne({ role: 'admin' });
    if (!adminExists) {
      const adminEmail = process.env.ADMIN_EMAIL || 'admin@balajitour.com';
      const adminPassword = process.env.ADMIN_PASSWORD || 'admin123';
      
      // Pass plain password - pre-save hook will hash it automatically
      await User.create({
        email: adminEmail,
        password: adminPassword, // Plain password - will be hashed by pre-save hook
        role: 'admin',
        name: 'Admin',
      });
      
      console.log('✅ Admin user created:', adminEmail);
    }
  } catch (error) {
    console.error('Error initializing admin:', error.message);
  }
};

const PORT = process.env.PORT || 5000;

app.listen(PORT, async () => {
  console.log(`🚀 Server running on port ${PORT}`);
  await initializeAdmin();
});

export default app;

