#!/bin/bash

# Script to update MongoDB password in server/.env

echo "🔐 MongoDB Password Updater"
echo "=========================="
echo ""

if [ ! -f "server/.env" ]; then
    echo "❌ server/.env file not found!"
    exit 1
fi

echo "Current MongoDB URI in server/.env:"
grep "MONGODB_URI" server/.env
echo ""

read -sp "Enter your MongoDB password: " PASSWORD
echo ""

if [ -z "$PASSWORD" ]; then
    echo "❌ Password cannot be empty!"
    exit 1
fi

# Escape special characters in password for sed
ESCAPED_PASSWORD=$(echo "$PASSWORD" | sed 's/[[\.*^$()+?{|]/\\&/g')

# Update the password in .env file
sed -i "s/<db_password>/$ESCAPED_PASSWORD/g" server/.env

echo "✅ Password updated in server/.env"
echo ""
echo "Updated MongoDB URI:"
grep "MONGODB_URI" server/.env | sed 's/:[^@]*@/:****@/'
echo ""
echo "🧪 Testing connection..."
echo ""

# Test connection
cd server
node -e "
import('dotenv').then(dotenv => {
  dotenv.default.config();
  import('mongoose').then(mongoose => {
    mongoose.default.connect(process.env.MONGODB_URI, { serverSelectionTimeoutMS: 5000 })
      .then(() => {
        console.log('✅ MongoDB connection successful!');
        process.exit(0);
      })
      .catch(err => {
        console.error('❌ MongoDB connection failed:', err.message);
        console.error('');
        console.error('💡 Check:');
        console.error('   1. Password is correct');
        console.error('   2. IP address is whitelisted in MongoDB Atlas');
        console.error('   3. Network access is enabled');
        process.exit(1);
      });
  });
});
" 2>/dev/null

if [ $? -eq 0 ]; then
    echo ""
    echo "🎉 Success! MongoDB is connected."
    echo ""
    echo "Next steps:"
    echo "  1. Seed database: cd server && npm run seed"
    echo "  2. Start server: npm run dev"
else
    echo ""
    echo "⚠️  Connection test failed. Please check your password and MongoDB Atlas settings."
fi

